/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <stdarg.h>
#include "cheops.h"



static int moncnt=0;
static GList *monitors=NULL;

struct monitor_list_row {
	struct net_object *no;
	struct monitor *mon;
	struct monitor_list_row *next;
	GtkWidget *label;
	GtkWidget *every;
	GtkWidget *entry;
	GtkWidget *units;
	GtkWidget *report;
	GtkWidget *error;
		GtkWidget *warning;
		GtkWidget *critical;
	GtkWidget *after;
	GtkWidget *failures;
	GtkWidget *ftext;
	GtkWidget *setup;
	void *setupdata;
};

struct monitor_box {
	GtkWidget *window;
	struct net_object *no;
	struct monitor_list_row *rows;
};


static void monitored_object(struct net_object *no, int level)
{

	/* Graphically display monitoring status */
	
	static GdkColor *yellow = NULL;
	static GdkColor *green = NULL;
	static GdkColor *red = NULL;
	static GdkColor *blue = NULL;

#ifndef USE_CANVAS
	static GtkStyle *yellow_s;
	static GtkStyle *green_s;
	static GtkStyle *red_s;
	static GtkStyle *blue_s;
#endif
	
	if (!blue) {
		blue = g_new0(GdkColor, 1);
		blue->red = 108 * 256;
		blue->green = 166 * 256;
		blue->blue = 205 * 256;
#ifdef USE_CANVAS
		gdk_color_alloc(gtk_widget_get_colormap(no->np->canvas), blue);
#else
		gdk_color_alloc(gdk_window_get_colormap(main_window.window->window), blue);
		blue_s = gtk_style_copy(gtk_widget_get_default_style());
		blue_s->bg[GTK_STATE_ACTIVE]=*blue;
		blue_s->bg[GTK_STATE_NORMAL]=*blue;
		blue_s->bg[GTK_STATE_SELECTED]=*blue;
		gtk_style_ref(blue_s);
#endif

	}
	
	if (!yellow) {
		yellow = g_new0(GdkColor, 1);
		yellow->red = 65535;
		yellow->green = 65535;
		yellow->blue = 0;
#ifdef USE_CANVAS
		gdk_color_alloc(gtk_widget_get_colormap(no->np->canvas), yellow);
#else
		gdk_color_alloc(gdk_window_get_colormap(main_window.window->window), yellow);
		yellow_s = gtk_style_copy(gtk_widget_get_default_style());
		yellow_s->bg[GTK_STATE_ACTIVE]=*yellow;
		yellow_s->bg[GTK_STATE_NORMAL]=*yellow;
		yellow_s->bg[GTK_STATE_SELECTED]=*yellow;
		gtk_style_ref(yellow_s);
#endif
	};
	
	if (!green) {
		green = g_new0(GdkColor, 1);
		green->red = 0;
		green->green = 205 * 256;
		green->blue = 0;
#ifdef USE_CANVAS
		gdk_color_alloc(gtk_widget_get_colormap(no->np->canvas), green);
#else
		gdk_color_alloc(gdk_window_get_colormap(main_window.window->window), green);
		green_s = gtk_style_copy(gtk_widget_get_default_style());
		green_s->bg[GTK_STATE_ACTIVE]=*green;
		green_s->bg[GTK_STATE_NORMAL]=*green;
		green_s->bg[GTK_STATE_SELECTED]=*green;
		gtk_style_ref(green_s);
#endif
	};
	
	if (!red) {
		red = g_new0(GdkColor, 1);
		red->red = 205 * 256;
		red->green = 0;
		red->blue = 0;
#ifdef USE_CANVAS
		gdk_color_alloc(gtk_widget_get_colormap(no->np->canvas), red);
#else
		gdk_color_alloc(gdk_window_get_colormap(main_window.window->window), red);
		red_s = gtk_style_copy(gtk_widget_get_default_style());
		red_s->bg[GTK_STATE_ACTIVE]=*red;
		red_s->bg[GTK_STATE_NORMAL]=*red;
		red_s->bg[GTK_STATE_SELECTED]=*red;
		gtk_style_ref(red_s);
#endif
	};
	
	switch(level) {
	case MONITOR_NOT_MONITORED:
#ifdef USE_CANVAS
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, NULL);
		gnome_canvas_item_set(no->rect, "fill_color","white", NULL);
		break;
	case MONITOR_MONITORED:
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, blue);
		gnome_canvas_item_set(no->rect, "fill_color", "skyblue3", NULL);
		break;
	case MONITOR_NOMINAL:
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, green);
		gnome_canvas_item_set(no->rect, "fill_color", "green3", NULL);
		break;
	case MONITOR_WARNING:
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, yellow);
		gnome_canvas_item_set(no->rect, "fill_color", "yellow", NULL);
		break;
	case MONITOR_CRITICAL:
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, red);
		gnome_canvas_item_set(no->rect, "fill_color", "red3", NULL);
		break;
#else
		gtk_widget_restore_default_style(no->otherbox);
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, NULL);
		break;
	case MONITOR_MONITORED:
		gtk_widget_set_style(no->otherbox, blue_s);
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, blue);
		break;
	case MONITOR_NOMINAL:
		gtk_widget_set_style(no->otherbox, green_s);
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, green);
		break;
	case MONITOR_WARNING:
		gtk_widget_set_style(no->otherbox, yellow_s);
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, yellow);
		break;
	case MONITOR_CRITICAL:
		gtk_widget_set_style(no->otherbox, red_s);
		gtk_clist_set_background(GTK_CLIST(no->np->list), no->id, red);
		break;
#endif
	default:
	}
}

static struct monitor_tag *mt_new(struct monitor *mon)
{
	/* Create and initialize a monitor tag */
	struct monitor_tag *mt;
	mt = g_new0(struct monitor_tag, 1);
	memset(mt, 0, sizeof(struct monitor_tag));
	mt->next = NULL;
	mt->mon = mon;
	mt->failures = 0;
	mt->tid = -1;
	mt->data = NULL;
	mt->id = -1;
	mt->lastmsg[0]='\0';
	mt->level = MONITOR_MONITORED;
	return mt;
}

struct monitor *str2mon(char *str)
{
	struct monitor *mon;
	GList *l;
	l = monitors;
	/* Find a monitor by its string representation */
	while(l) {
		mon = (struct monitor *)(l->data);
		if (!strcmp(str, mon->service))
			break;
		l = g_list_next(l);
	}
	if (!l)
		mon=NULL;
	return mon;
}

static void monitor_save(struct net_object *no)
{
	char buf[256];
	char *c;
	struct in_addr ia;
	struct monitor_tag *mt;
	FILE *f;
	/* Save the monitoring status of a host to a file */
	if (!(c = getenv("HOME")))
		return;
	snprintf(buf, sizeof(buf), "%s/.cheops/monitors",c);
	mkdir(buf, 0700);
	ia.s_addr = no->ip_addr;
	snprintf(buf, sizeof(buf), "%s/.cheops/monitors/%s", c, inet_ntoa(ia));
	if (no->mon) {
		f = fopen(buf, "w");
		if (f) {
			mt = no->mon;
			while(mt) {
				fprintf(f, "[%s^%d^%d^%d^%s]\n",
					mt->mon->service,
					mt->error_level,
					mt->maxfailures,
					mt->frequency,
					((mt->setupdata && mt->mon->setup2str) ? 
						mt->mon->setup2str(mt->setupdata) :
						""));
				mt=mt->next;
			}
		}
		fclose(f);
	} else 
		unlink(buf);
}

static void set_monitor_state(struct net_object *no)
{
	/* Calculate and display the monitor status */
	int state = MONITOR_NOT_MONITORED;
	struct monitor_tag *mt;
	mt = no->mon;
	while(mt) {
		/* Scroll through all the services and find the highest event level */
#if 0
		if (mt->failures >= mt->maxfailures)
			mt->level = mt->error_level;
#endif
		if (mt->level > state)
			state = mt->level;
		mt=mt->next;
	}
	/* Update the graphic visually */
	monitored_object(no, state);
}

static int do_monitor(void *);

static int monitor_timeout(void *data)
{
	/* The service handler hasn't gotten back to us in time.  We log a timeout
	   but the service handler can still log a success if it eventually does
	   get through */
	struct monitor_tag *mt = (struct monitor_tag *)data;
	mt->tid = -1;
	if (mt->mon->timeout_callback) {
		mt->mon->timeout_callback(mt->no, &(mt->data), ((mt->failures + 1) < mt->maxfailures) ? 0 : 1);
	} else {
		monitor_report(mt->no, mt->tonp, mt->error_level, mt->mon->service, "Timeout");
	}
	if (option_immediate_retry)
		do_monitor(data);
	return FALSE;
}




void monitor_report(struct net_object *no, struct net_page *np, int status, char *svc, char *msg)
{
	/* Called by timeout or by connection to log a report */
	struct monitor_tag *mt;
	
	/* Check validity of the object, since this is being made from a callback */
	if (!valid_no(np, no)) {
		fprintf(stderr, "Warning: bad monitor report!\n");
		return;
	}
	
	/* Find the monitor entry for this service */
	mt = no->mon;
	while(mt) {
		if (!strcmp(mt->mon->service, svc)) {
			/* Cancel any timeouts */
			if (mt->tid > -1) {
				gtk_timeout_remove(mt->tid);
				mt->tid = -1;
			}
			/* Reset the failure count if this is a nominal */
			if (status == MONITOR_NOMINAL) {
				mt->failures=0;
				if (mt->level < MONITOR_WARNING)  {
					/* And leave if we're not currently in error state */
					mt->level = status;
					goto out;
				}
			}
			if (status > MONITOR_NOMINAL) {
				/* If there is an event, we increment the failure pointer */
				mt->failures++;
				/* But stop with that unless we've at least equaled the # of allowed
				   failures.  Don't update status either */
				if (mt->failures < mt->maxfailures)
					goto out;
				/* If we're already in an error condition, then only log
				   if something changes */
				if ((mt->level == status) && !strcmp(mt->lastmsg, msg))
					goto out;
				/* Something bad has happend.  If the user has selected it
				   we should be sure this page has the focus */
				select_page(np);
			}
			/* Change levels, and log an entry */
			mt->level = status;
			strncpy(mt->lastmsg, msg, sizeof(mt->lastmsg));
			
			log_event(status, "monitor", svc, no->hostname, msg);
out:
			set_monitor_state(no);
			return;
		}
		mt = mt->next;
	}
	fprintf(stderr, "Warning: unable to find tag for service %s on host %s\n", svc, no->hostname);
}

static void setup_do(GtkWidget *w)
{
	/* When they click on the setup button, replace the setupdata with the
	   newly allocated setup data retrieved from the setup procedure */
	struct monitor_list_row *mlr = 
		(struct monitor_list_row *)gtk_object_get_user_data(GTK_OBJECT(w));
	if (mlr->mon->setup) 
		mlr->setupdata = mlr->mon->setup(mlr->setupdata, mlr->no);
}

static void set_active_row(GtkWidget *w)
{
	/* Simply activate the necessary widgets when
	   the user clicks to enable a given monitor.  
	   We only enable the setup button if there is
	   a setup routine for the service */
	int tmp = GTK_TOGGLE_BUTTON(w)->active;
	struct monitor_list_row *mlr = 
		(struct monitor_list_row *)gtk_object_get_user_data(GTK_OBJECT(w));
	gtk_widget_set_sensitive(mlr->every, tmp);
	gtk_widget_set_sensitive(mlr->entry, tmp);
	gtk_widget_set_sensitive(mlr->units, tmp);
	gtk_widget_set_sensitive(mlr->report, tmp);
	gtk_widget_set_sensitive(mlr->error, tmp);
	gtk_widget_set_sensitive(mlr->after, tmp);
	gtk_widget_set_sensitive(mlr->failures, tmp);
	gtk_widget_set_sensitive(mlr->ftext, tmp);
	if (mlr->mon->setup)
		gtk_widget_set_sensitive(mlr->setup, tmp);
	
}

#if 0
static void update_line(GtkWidget *w)
{
	char *old;
	char *new;
	struct monitor_list_row *mlr = 
		(struct monitor_list_row *)gtk_object_get_user_data(GTK_OBJECT(w));
	if (gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(mlr->failures)) == 1) 
		new = " failure  ";
	else
		new = " failures  ";
	gtk_label_get(GTK_LABEL(mlr->ftext), &old);
	if (strcmp(old, new))
		gtk_label_set(GTK_LABEL(mlr->ftext), new);
}
#endif

static struct monitor_tag *mlr2mt(struct monitor_list_row *mlr)
{
	/* This routine converts a row into a monitor tag.  Some
	   fields that we don't know are left in an unknown state,
	   but the parent procedure properly fills them in.  If the 
	   row isn't available, we discard any setupdata that is
	   in it */
	   
	struct monitor_tag *mt=NULL;
	GtkWidget *menuitem;
	if (GTK_TOGGLE_BUTTON(mlr->label)->active) {
		menuitem = GTK_OPTION_MENU(mlr->error)->menu_item;
		mt = mt_new(mlr->mon);
		mt->setupdata = mlr->setupdata;
#if 0
		/* This should work, why doesn't it?  the child is always nil :((( */
		printf("menuitem is %p\n", menuitem);
		label = GTK_BIN(menuitem)->child;
		printf("label is %p\n", label);
#else
		if (menuitem == mlr->warning)
			mt->error_level = MONITOR_WARNING;
		else if (menuitem == mlr->critical)
			mt->error_level = MONITOR_CRITICAL;
		else
			fprintf(stderr, "What did you do???\n");
#endif
		mt->maxfailures = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(mlr->failures));
		mt->frequency = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(mlr->entry)) * 60;
	} else {
		if (mlr->setupdata) {
			if (mlr->mon->setup_cleanup)
				mlr->mon->setup_cleanup(mlr->setupdata);
			else
				g_free(mlr->setupdata);
		}
	}
	return mt;
}

static int fudge()
{
	/* Generate a random fudge factor of +/- 5000 ms to prevent all events
	   from happening at exactly the same time. */
	int res;
	res= (int)((double)rand()/(double)RAND_MAX * (double)10000) - 5000;
	return res;
}

static int schedule_monitor(struct monitor_tag *mt, int now)
{
	/* Schedule a monitoring run to be done sometime
	   in the future (maybe now), including ourselves.
	   add a fudge factor to prevent all the services
	   from being activated at exactly the same time. */
	if (mt->id > -1)
		gtk_timeout_remove(mt->id);
	mt->id = gtk_timeout_add((now ? 5 : mt->frequency) * 1000 + fudge(), do_monitor, mt);
	return TRUE;
}

static int do_monitor(void *data)
{
	/* Actually monitor a host */
	struct monitor_tag *mt;
	mt = (struct monitor_tag *)data;
	
	/* Schedule the next event */
	schedule_monitor(mt, 0);
	
	/* Schedule the next timeout, in case the handler doesn't give us a reply in time */
	if (mt->tid > -1)
		gtk_timeout_remove(mt->tid);
	mt->tid = gtk_timeout_add(connection_timeout * 1000, monitor_timeout, mt);
	
	/* Get the ball rolling */
	mt->mon->monitor(mt->no, mt->error_level, &(mt->data), mt->setupdata);
	return TRUE;
}

/* we proxy the callbacks to make then one-time only runs */

struct callback_proxy {
	struct callback_proxy *next;
	int id;
	void *data;
	void (*callback)(void *, int, GdkInputCondition);
};

struct callback_proxy *cpq=NULL;

struct callback_proxy *cp_unlink(int id)
{
	struct callback_proxy *cp = cpq, *prev=NULL;

	/* Find proxy with matching ID, unlink it, and return
	   the now isolated structure */
	while(cp) {
		if (cp->id == id) {
			if (prev) {
				prev->next= cp->next;
			} else {
				cpq = cp->next;
			}
			cp->next = NULL;
			break;
		}
		prev=cp;
		cp=cp->next;
	}
	return cp;
}

void cancel_data(int id)
{
	/* Just cancel any outstnading id */
	struct callback_proxy *cp = cp_unlink(id);
	if (!cp) {
		fprintf(stderr, "Unable to locate callback proxy to cancel!\n");
	} else {
		gdk_input_remove(id);
		g_free(cp);
	}
}

static void data_is_ready(void *data, int s, GdkInputCondition c)
{
	struct callback_proxy *cp, *cp2 = (struct callback_proxy *)data;
	/* Theoretically, data is our pointer, but we validate it
	   to be sure anyway */
	cp = cp_unlink(cp2->id);
	if (!cp) {
		/* Generate a crash */
		printf("Unable to find proxy for %d\n", cp2->id);
		*(int *)(NULL)=1;
	} else if (cp != cp2) {
		fprintf(stderr, "Disagreement about callback proxies!\n");
	} else {
		gdk_input_remove(cp->id); 
#ifdef DEBUG
		printf("callback to call is %p\n", cp->callback);
#endif
		cp->callback(cp->data, s, c);
		g_free(cp);
	}
}

int wait_for_data(int s, void (*callback)(void *, int, GdkInputCondition), void *data)
{
	struct callback_proxy *cp;
	
	/* Make and populate a proxy structure, waiting
	   for input or an exception */
	cp = g_new0(struct callback_proxy, 1);
	cp->data = data;
	cp->callback = callback;
	cp->id = gdk_input_add(s, GDK_INPUT_READ | GDK_INPUT_EXCEPTION, data_is_ready, cp);
	cp->next = cpq;
	cpq = cp;
	return cp->id;
}

int try_to_connect(int s, unsigned int ip, unsigned short port, 
			  void *data, void (*callback)(void *, int, GdkInputCondition)) 
{
	int flags;
	struct sockaddr_in sin;
	struct callback_proxy *cp;
	
	cp = g_new0(struct callback_proxy, 1);
	/* Make socket non-blocking */
	flags = fcntl(s, F_GETFL);
	flags |= O_NONBLOCK;
	fcntl(s, F_SETFL, flags);
	sin.sin_addr.s_addr = ip;
	sin.sin_port = port;
	sin.sin_family = AF_INET;
	connect(s, &sin, sizeof(sin));
	/* Populate and enqueue proxy structure */
	cp->data = data;
	cp->callback = callback;
	cp->id = gdk_input_add(s, GDK_INPUT_WRITE | GDK_INPUT_READ | GDK_INPUT_EXCEPTION, data_is_ready, cp);
	cp->next = cpq;
	cpq = cp;
	return cp->id;
	
}

void destroy_tag(struct monitor_tag *mt)
{
	/* This is the monitor timer */
	if (mt->id > -1)
		gtk_timeout_remove(mt->id);
	/* Here we destroy the timeout timer */
	if (mt->tid > -1)
			gtk_timeout_remove(mt->tid);
	/* And here we free any data that is used by the tag */
	if (mt->data) {
		if (mt->mon->cleanup)
			mt->mon->cleanup(&mt->data);
		else
			fprintf(stderr, "Unable to cleanup monitor tag,  service %s, host %s\n", mt->mon->service, mt->no->hostname);
	}
	g_free(mt);
}

static void close_monitor_box(GtkWidget *w)
{
	struct monitor_box *mb =
		(struct monitor_box *)gtk_object_get_user_data(GTK_OBJECT(w));
	struct monitor_tag *mt, *mt2;
	struct monitor_list_row *mlr, *mlr2;
	mt = mb->no->mon;
	while(mt) {
		mt2 = mt;
		mt = mt->next;
		destroy_tag(mt2);
	};
	mb->no->mon = NULL;
	mlr = mb->rows;
	
	while(mlr) {
		mlr2 = mlr;
		if ((mt = mlr2mt(mlr))) {
			mt->no = mb->no;
			mt->tonp = mb->no->np;
			mt->next = mt->no->mon;
			mb->no->mon = mt;
			do_monitor(mt);
		}
		mlr = mlr->next;
		g_free(mlr2);
	}
	gtk_widget_destroy(mb->window);
	set_monitor_state(mb->no);
	g_free(mb);
}

static void close_save_monitor_box(GtkWidget *w)
{
	struct monitor_box *mb =
		(struct monitor_box *)gtk_object_get_user_data(GTK_OBJECT(w));
	struct net_object *no = mb->no;
	close_monitor_box(w);
	monitor_save(no);
}

static void update_mlr_tag(struct monitor_tag *mt, struct monitor_list_row *mlr)
{
	/* This actually does alot of the work here: */
	mlr->setupdata = mt->setupdata;
	mt->setupdata = NULL;
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(mlr->label), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(mlr->entry), (double)(mt->frequency)/60.0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(mlr->failures), (double)(mt->maxfailures));
	gtk_option_menu_set_history(GTK_OPTION_MENU(mlr->error), mt->error_level - MONITOR_WARNING);
	
	
}

int get_socket_error(int fd)
{
	int res=0;
	int reslen;
	reslen = sizeof(res);
	if (getsockopt(fd, SOL_SOCKET, SO_ERROR, &res, &reslen) < 0)
		fprintf(stderr, "getsockopt() failed: %s\n", strerror(errno));
	return res;
}

static struct monitor_list_row *show_add_monitor(GtkTable *table, int x, struct monitor *m)
{
	struct monitor_list_row *mlr;
	GtkObject *adj;
	GtkWidget *mi;
	GtkWidget *menu;
	mlr = g_new0(struct monitor_list_row, 1);
	mlr->mon = m;
	/* Webserver  (1) */
	mlr->label = gtk_check_button_new_with_label(m->label);
	gtk_object_set_user_data(GTK_OBJECT(mlr->label), mlr);
	gtk_signal_connect(GTK_OBJECT(mlr->label), "clicked", GTK_SIGNAL_FUNC(set_active_row), mlr);
	gtk_table_attach_defaults(table, mlr->label, 0, 1, x, x+1);
	gtk_widget_show(mlr->label);
	
	/* Every (2) */
	mlr->every = gtk_label_new(" every ");
	gtk_table_attach_defaults(table, mlr->every, 1, 2, x, x+1);
	gtk_widget_set_sensitive(mlr->every, FALSE);
	gtk_widget_show(mlr->every);
	
	/* 5 (3) */
	adj = gtk_adjustment_new(5.0, 1.0, 60.0, 1.0, 1.0, 1.0);
	mlr->entry = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 5.0, 0);
	gtk_table_attach_defaults(table, mlr->entry, 2, 3, x, x+1);
	gtk_widget_set_sensitive(mlr->entry, FALSE);
	gtk_widget_show(mlr->entry);
	
	/* Minutes/Seconds (eventually to be a combo)  (4) */
	mlr->units = gtk_label_new(" minute(s) ");
	gtk_table_attach_defaults(table, mlr->units, 3, 4, x, x+1);
	gtk_widget_set_sensitive(mlr->units, FALSE);
	gtk_widget_show(mlr->units);
	
	/* Reporting (5) */
	mlr->report = gtk_label_new(", reporting ");
	gtk_table_attach_defaults(table, mlr->report, 4, 5, x, x+1);
	gtk_widget_set_sensitive(mlr->report, FALSE);
	gtk_widget_show(mlr->report);
	
	/* Critical (6) */
	mlr->error = gtk_option_menu_new();
	menu = gtk_menu_new();
	mlr->warning = mi = gtk_menu_item_new_with_label("Warning");
	gtk_widget_show(mi);
	gtk_menu_append(GTK_MENU(menu), mi);
	mlr->critical = mi = gtk_menu_item_new_with_label("Critical");
	gtk_widget_show(mi);
	gtk_menu_append(GTK_MENU(menu), mi);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(mlr->error), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(mlr->error), m->default_level - MONITOR_WARNING);
	
	gtk_table_attach_defaults(table, mlr->error, 5, 6, x, x+1);
	gtk_widget_set_sensitive(mlr->error, FALSE);
	gtk_widget_show(mlr->error);
	
	/* Event after (7) */
	mlr->after = gtk_label_new(" event after ");
	gtk_table_attach_defaults(table, mlr->after, 6, 7, x, x+1);
	gtk_widget_set_sensitive(mlr->after, FALSE);
	gtk_widget_show(mlr->after);
	
	adj = gtk_adjustment_new(1.0, 1.0, 20.0, 1.0, 1.0, 1.0);
	
	/* 1  (8) */
	mlr->failures = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1.0, 0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(mlr->failures), 1.0);
	gtk_table_attach_defaults(table, mlr->failures, 7, 8, x, x+1);
	gtk_widget_set_sensitive(mlr->failures, FALSE);
	gtk_widget_show(mlr->failures);
	
	/* failure(s) (9) */
	mlr->ftext = gtk_label_new(" failure(s)  ");
	gtk_table_attach_defaults(table, mlr->ftext, 8, 9, x, x+1);
	gtk_widget_set_sensitive(mlr->ftext, FALSE);
	gtk_widget_show(mlr->ftext);
	
	gtk_object_set_user_data(GTK_OBJECT(mlr->label), mlr);
	
	mlr->setup = gtk_button_new_with_label("  Setup...  ");
	gtk_table_attach_defaults(table, mlr->setup, 9, 10, x, x+1);
	gtk_widget_set_sensitive(mlr->setup, FALSE);
	gtk_widget_show(mlr->setup);
	
	gtk_object_set_user_data(GTK_OBJECT(mlr->setup), mlr);
	gtk_signal_connect(GTK_OBJECT(mlr->setup), "clicked", GTK_SIGNAL_FUNC(setup_do), mlr);
	mlr->setupdata = NULL;
	return mlr;
}

void show_monitor_box(struct net_object *no) {
	GtkWidget *vbox;
	GtkWidget *table;
	GtkWidget *bbox;
	GtkWidget *close;
	GtkWidget *save;
	GtkWidget *label;
	char buf[256];
	struct monitor_box *mb;
	struct monitor *m;
	struct monitor_list_row *mlr;
	struct monitor_tag *mt;
	GList *l;
	int x;
	mb = g_new0(struct monitor_box, 1);
	mb->no = no;
	mb->rows = NULL;
	mb->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_realize(mb->window);
	fix_icon(mb->window->window);
	
	g_snprintf(buf, sizeof(buf), "Monitor %s: ", no->hostname);
	label = gtk_label_new(buf);
	gtk_widget_show(label);
	gtk_window_set_title(GTK_WINDOW(mb->window), buf);
	
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(vbox);
	
	table = gtk_table_new(moncnt, 10, FALSE);
	gtk_widget_show(table);
	
	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(bbox);
	
	l=monitors;
	for (x=0;x<moncnt;x++) {
		if (l) {
			m = (struct monitor *)(l->data);
			if (m) {
				mlr = show_add_monitor(GTK_TABLE(table), x, m);
				mlr->no = no;
				mlr->next = mb->rows;
				mb->rows = mlr;
				mt = no->mon;
				while(mt) {
					if (mt->mon == m) 
						update_mlr_tag(mt, mlr);
					mt = mt->next;
				}
			}
		}
		l = g_list_next(l);
	}
	
	close = gtk_button_new_with_label("  Close  ");
	save = gtk_button_new_with_label("  Save  ");
	gtk_widget_show(close);
	gtk_widget_show(save);
	
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox), close, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox), save, FALSE, FALSE, 5);
	
	gtk_container_add(GTK_CONTAINER(mb->window), vbox);
	gtk_object_set_user_data(GTK_OBJECT(mb->window), mb);
	gtk_object_set_user_data(GTK_OBJECT(close), mb);
	gtk_object_set_user_data(GTK_OBJECT(save), mb);

	gtk_signal_connect(GTK_OBJECT(mb->window), "delete_event", GTK_SIGNAL_FUNC(close_monitor_box), NULL);
	gtk_signal_connect(GTK_OBJECT(close), "clicked", GTK_SIGNAL_FUNC(close_monitor_box), NULL);
	gtk_signal_connect(GTK_OBJECT(save), "clicked", GTK_SIGNAL_FUNC(close_save_monitor_box), NULL);
	gtk_widget_show(mb->window);
}

static int portsort(const void *a, const void *b)
{
	struct monitor *m1, *m2;
	m1 = (struct monitor *)a;
	m2 = (struct monitor *)b;
	return strcmp((m1)->service, (m2)->service);
}

void init_monitoring()
{
	GList *l;
	struct monitor *m;
	int (*get_default)();
	l = list_plugin(TYPE_MONITOR);
	while(l) {
		m = g_new0(struct monitor, 1);
		m->monitor = get_plugin((char *)l->data, TYPE_MONITOR, MONITOR_MONITOR);
		if (m->monitor) {
			get_default = (int (*)())get_plugin((char *)l->data, TYPE_MONITOR, MONITOR_DEFAULT_LEVEL);
			if (get_default) {
				m->default_level = get_default();
				/* Optional things */
				m->cleanup = get_plugin((char *)l->data, TYPE_MONITOR, MONITOR_CLEANUP);
				m->setup = get_plugin((char *)l->data, TYPE_MONITOR, MONITOR_SETUP);
				m->setup_cleanup = get_plugin((char *)l->data, TYPE_MONITOR, MONITOR_SETUP_CLEANUP);
				m->setup2str = get_plugin((char *)l->data, TYPE_MONITOR, MONITOR_SETUP_STR);
				m->str2setup = get_plugin((char *)l->data, TYPE_MONITOR, MONITOR_STR_SETUP);
				m->timeout_callback = get_plugin((char *)l->data, TYPE_MONITOR, MONITOR_TIMEOUT_CB);
				strncpy(m->service, (char *)l->data, sizeof(m->service));
				strncpy(m->label, label_plugin(m->service), sizeof(m->label));
				monitors = g_list_insert_sorted(monitors, m, portsort);
				moncnt++;
			} else
				fprintf(stderr, "Warning: monitor %s does not have a default level function\n", (char *)l->data);
			
		} else 
			fprintf(stderr, "Warning: monitor %s does not have a monitor function\n", (char *)l->data);
		l = g_list_next(l);
	}
	g_list_free(l);
	srand(time(NULL));
}

void monitor_update(struct net_object *no)
{
	char buf[1024];
	char *c;
	struct in_addr ia;
	struct monitor_tag *mt;
	struct monitor *mon;
	FILE *f;
	if (!(c = getenv("HOME")))
		return;
	ia.s_addr = no->ip_addr;
	snprintf(buf, sizeof(buf), "%s/.cheops/monitors/%s", c, inet_ntoa(ia));
	f = fopen(buf, "r");
	if (!f)
		return;
	while(!feof(f)) {
		fgets(buf, sizeof(buf), f);
		if (!feof(f)) {
			c = buf+1;
			c = strtok(c, "^");
			mon = str2mon(c);
			if (mon) {
				mt = mt_new(mon);
				mt->no = no;
				mt->tonp = no->np;
				c = strtok(NULL, "^");
				if (c)
					mt->error_level = atoi(c);
				else
					mt->error_level = mon->default_level;
				c = strtok(NULL, "^");
				if (c)
					mt->maxfailures = atoi(c);
				else
					mt->maxfailures = 1;
				c = strtok(NULL, "^");
				if (c)
					mt->frequency = atoi(c);
				else
					mt->frequency = 60;
				c = strtok(NULL, "^");
				/* Trim off trailing ]\n */
				c[strlen(c) - 2]='\0';
				mt->setupdata = NULL;
				if (strlen(c)) {
					if (mon->str2setup)
						mt->setupdata = mon->str2setup(c, no);
				}
				mt->next = no->mon;
				no->mon = mt;
				schedule_monitor(mt, 1);
			} else {
				fprintf(stderr, "Warning: unable to find monitor '%s' on host '%s'\n", c, no->hostname);
			}
		}
	}
	set_monitor_state(no);
	fclose(f);
}

