#include "agent-nmapxml.h"

//#define DEBUG_NMAPXML

#ifdef DEBUG_NMAPXML
	#define DEBUG(x) x
#else
	#define DEBUG(a)
#endif

void g_free_not_null(void *it)
{
	if(it)
		g_free(it);
}

void nmapPortFreeEntry(nmap_host_port_t *p, void *arg)
{
	g_free_not_null(p);
}

void nmap_scan_free(nmap_scan_t *n)
{
	if(n)
	{
		g_free_not_null(n->scanner);
		g_free_not_null(n->args);
		g_free_not_null(n->version);
		g_free_not_null(n->xmloutputversion);

		g_free_not_null(n->host.addr);
		g_list_foreach(n->host.ports, (GFunc)nmapPortFreeEntry, NULL);
		g_list_free(n->host.ports);
		g_free_not_null(n->host.uptime_seconds);
		g_free_not_null(n->host.uptime_last_boot);

		g_free_not_null(n->os.name);
		g_free_not_null(n->os.accuracy);
	}
}

static void parseNmapScanPortService(nmap_host_port_t *port, xmlNode *node)
{
	xmlAttr *prop = node->properties;

	DEBUG(printf("%s()\n", __FUNCTION__));
	while(prop)
	{
		if(prop->type == XML_ATTRIBUTE_NODE)
		{
			if(0 == xmlStrcmp(prop->name, (const xmlChar *) "name"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(port->service)
						g_free(port->service);
					port->service = g_strdup(prop->val->content);
				}
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "proto"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(0 == xmlStrcmp(prop->val->content, "rpc"))
						port->proto = PORT_PROTO_RPC;
				}
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "rpcnum"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
					port->rpcnum = atoi(prop->val->content);
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "lowver"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
					port->rpclowver = atoi(prop->val->content);
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "highver"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
					port->rpchighver = atoi(prop->val->content);
			}
		}
		prop = prop->next;
	}
}

static void parseNmapScanPortState(nmap_host_port_t *port, xmlNode *node)
{
	xmlAttr *prop = node->properties;

	DEBUG(printf("%s()\n", __FUNCTION__));
	while(prop)
	{
		if(prop->type == XML_ATTRIBUTE_NODE)
		{
			if(0 == xmlStrcmp(prop->name, (const xmlChar *) "state"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(0 == xmlStrcmp(prop->val->content, "open"))
						port->state = PORT_STATE_OPEN;
					else if(0 == xmlStrcmp(prop->val->content, "closed"))
						port->state = PORT_STATE_CLOSED;
					else if(0 == xmlStrcmp(prop->val->content, "filtered"))
						port->state = PORT_STATE_FILTERED;
					else if(0 == xmlStrcmp(prop->val->content, "UNfilitered"))
						port->state = PORT_STATE_UNFILTERED;
					else if(0 == xmlStrcmp(prop->val->content, "unknown"))
						port->state = PORT_STATE_UNKNOWN;
				}
			}
		}
		prop = prop->next;
	}
}

static void parseNmapScanPortOwner(nmap_host_port_t *port, xmlNode *node)
{
	xmlAttr *prop = node->properties;

	DEBUG(printf("%s()\n", __FUNCTION__));
	while(prop)
	{
		if(prop->type == XML_ATTRIBUTE_NODE)
		{
			if(0 == xmlStrcmp(prop->name, (const xmlChar *) "name"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(port->owner)
						g_free(port->owner);
					port->owner = g_strdup(prop->val->content);
				}
			}
		}
		prop = prop->next;
	}
}

static void parseNmapScanPort(nmap_scan_t *nms, xmlNode *node)
{
	xmlAttr *prop = node->properties;
	nmap_host_port_t *port = malloc(sizeof(*port));
	
	DEBUG(printf("%s()\n", __FUNCTION__));
	if(port == NULL)
	{
		perror("out of memory");
		exit(-1);
	}
	memset(port, 0, sizeof(*port));
	
	while(prop)
	{
		if(prop->type == XML_ATTRIBUTE_NODE)
		{
			if(0 == xmlStrcmp(prop->name, (const xmlChar *) "protocol"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(0 == xmlStrcmp(prop->val->content, "tcp"))
						port->protocol = PORT_PROTOCOL_TCP;
					else if(0 == xmlStrcmp(prop->val->content, "udp"))
						port->protocol = PORT_PROTOCOL_UDP;
					else if(0 == xmlStrcmp(prop->val->content, "ip"))
						port->protocol = PORT_PROTOCOL_IP;
				}
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "portid"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
					port->portid = atoi(prop->val->content);
			}
		}
		prop = prop->next;
	}

	/*
	 * Now, walk the tree.
	 */
	node = node->xmlChildrenNode;
	while (node) 
	{
		if(!xmlIsBlankNode(node))
		{
			if(0 == xmlStrcmp(node->name, (xmlChar *)"service"))
				parseNmapScanPortService(port, node);
			else if(0 == xmlStrcmp(node->name, (xmlChar *)"state"))
				parseNmapScanPortState(port, node);
			else if(0 == xmlStrcmp(node->name, (xmlChar *)"owner"))
				parseNmapScanPortOwner(port, node);
		}
		node = node->next;
	}
	
	nms->host.ports = g_list_prepend(nms->host.ports, port);
}

static void parseNmapScanPorts(nmap_scan_t *nms, xmlNode *node)
{
	DEBUG(printf("%s()\n", __FUNCTION__));
	/*
	 * Now, walk the tree.
	 */
	node = node->xmlChildrenNode;
	while (node) 
	{
		if(!xmlIsBlankNode(node))
		{
			if(0 == xmlStrcmp(node->name, (xmlChar *)"port"))
				parseNmapScanPort(nms, node);
		}
		node = node->next;
	}
}

static void parseNmapScanOSosmatch(nmap_scan_t *nms, xmlNode *node)
{
	xmlAttr *prop = node->properties;

	DEBUG(printf("%s()\n", __FUNCTION__));
	while(prop)
	{
		if(prop->type == XML_ATTRIBUTE_NODE)
		{
			if(0 == xmlStrcmp(prop->name, (const xmlChar *) "name"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(nms->os.name)
						g_free(nms->os.name);
					nms->os.name = g_strdup(prop->val->content);
				}
			}
		}
		prop = prop->next;
	}
}

static void parseNmapScanOS(nmap_scan_t *nms, xmlNode *node)
{
	DEBUG(printf("%s()\n", __FUNCTION__));
	/*
	 * Now, walk the tree.
	 */
	node = node->xmlChildrenNode;
	while (node) 
	{
		if(!xmlIsBlankNode(node))
		{
			if(0 == xmlStrcmp(node->name, (xmlChar *)"osmatch"))
				parseNmapScanOSosmatch(nms, node);
		}
		node = node->next;
	}
}

static void parseNmapScanAddress(nmap_scan_t *nms, xmlNode *node)
{
	xmlAttr *prop = node->properties;

	DEBUG(printf("%s()\n", __FUNCTION__));
	while(prop)
	{
		if(prop->type == XML_ATTRIBUTE_NODE)
		{
			if(0 == xmlStrcmp(prop->name, (const xmlChar *) "addr"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(nms->host.addr)
						g_free(nms->host.addr);
					nms->host.addr = g_strdup(prop->val->content);
				}
			}
		}
		prop = prop->next;
	}
}

static void parseNmapScanUptime(nmap_scan_t *nms, xmlNode *node)
{
	xmlAttr *prop = node->properties;

	DEBUG(printf("%s()\n", __FUNCTION__));
	while(prop)
	{
		if(prop->type == XML_ATTRIBUTE_NODE)
		{
			if(0 == xmlStrcmp(prop->name, (const xmlChar *) "seconds"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(nms->host.uptime_seconds)
						g_free(nms->host.uptime_seconds);
					nms->host.uptime_seconds = g_strdup(prop->val->content);
				}
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "lastboot"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
				{
					if(nms->host.uptime_last_boot)
						g_free(nms->host.uptime_last_boot);
					nms->host.uptime_last_boot = g_strdup(prop->val->content);
				}
			}
		}
		prop = prop->next;
	}
}

static void parseNmapScanHost(nmap_scan_t *nms, xmlNode *node)
{
	DEBUG(printf("%s()\n", __FUNCTION__));
	/*
	 * Now, walk the tree.
	 */
	node = node->xmlChildrenNode;
	while (node) 
	{
		if(!xmlIsBlankNode(node))
		{
			if(0 == xmlStrcmp(node->name, (xmlChar *)"ports"))
				parseNmapScanPorts(nms, node);
			else if(0 == xmlStrcmp(node->name, (xmlChar *)"os"))
				parseNmapScanOS(nms, node);
			else if(0 == xmlStrcmp(node->name, (xmlChar *)"address"))
				parseNmapScanAddress(nms, node);
			else if(0 == xmlStrcmp(node->name, (xmlChar *)"uptime"))
				parseNmapScanUptime(nms, node);
		}
		node = node->next;
	}
}

static nmap_scan_t *parseNmapScan(xmlDoc *doc)
{
	nmap_scan_t *ret;
	xmlNode *cur;
	xmlAttr *prop;
	
	DEBUG(printf("%s()\n", __FUNCTION__));
	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		fprintf(stderr, "empty document\n");
		xmlFreeDoc(doc);
		return (NULL);
	}

	if (xmlStrcmp(cur->name, (const xmlChar *) "nmaprun")) {
		fprintf(stderr, "document of the wrong type, root node != nmaprun but '%s'", cur->name);
		xmlFreeDoc(doc);
		return (NULL);
	}

	/*
	 * Allocate the structure to be returned.
	 */
	ret = (nmap_scan_t *)malloc(sizeof(*ret));
	if (ret == NULL) {
		fprintf(stderr, "out of memory\n");
		xmlFreeDoc(doc);
		return (NULL);
	}
	memset(ret, 0, sizeof(*ret));


	prop = cur->properties;
	while(prop)
	{
		if(prop->type == XML_ATTRIBUTE_NODE)
		{
			if(0 == xmlStrcmp(prop->name, (const xmlChar *) "xmloutputversion"))
			{
				
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "version"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
					ret->version = g_strdup(prop->val->content);
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "args"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
					ret->args = g_strdup(prop->val->content);
			}
			else if(0 == xmlStrcmp(prop->name, (const xmlChar *) "scanner"))
			{
				if(prop->val && prop->val->type == XML_TEXT_NODE)
					ret->scanner = g_strdup(prop->val->content);
			}
		}
		prop = prop->next;
	}

	/*
	 * Now, walk the tree.
	 */
	/* First level we expect just Jobs */
	cur = cur->xmlChildrenNode;
	while (cur) 
	{
		if(!xmlIsBlankNode(cur))
		{
			if(0 == xmlStrcmp(cur->name, (xmlChar *)"host"))
				parseNmapScanHost(ret, cur);
		}
		cur = cur->next;
	}
	
	xmlFreeDoc(doc);

	return (ret);
}

void printPorts(nmap_host_port_t *port, void *arg)
{
	if(port)
	{
		printf("host port protocol = %s\n", (port->protocol == PORT_PROTOCOL_TCP ? "tcp" : 
		                                     (port->protocol == PORT_PROTOCOL_UDP ? "udp": 
		                                      (port->protocol == PORT_PROTOCOL_IP ? "ip": "unknown"))));
		printf("host port portid = %d\n", port->portid);
		printf("host port state = %s\n", (port->state == PORT_STATE_OPEN ? "open" :
		                                  (port->state == PORT_STATE_CLOSED ? "closed" :
		                                   (port->state == PORT_STATE_FILTERED ? "filtered" :
		                                    (port->state == PORT_STATE_UNFILTERED ? "UNfiltered" : "unknown")))));
		printf("host port service = %s\n", port->service);
		printf("host port owner = %s\n", port->owner);
		printf("host port rpcnum = %d\n", port->rpcnum);
		printf("host port rpclowver = %d\n", port->rpclowver);
		printf("host port rpchighver = %d\n", port->rpchighver);
	}
}

void nmap_scan_print(nmap_scan_t *cur)
{
	printf("scanner = %s\n", cur->scanner);
	printf("args = %s\n", cur->args);
	printf("version = %s\n", cur->version);

	printf("host addr = %s\n", cur->host.addr);
	g_list_foreach(cur->host.ports, (GFunc)printPorts, NULL);
	
	printf("host os name = %s\n", cur->os.name);
}

xmlDoc *runNmapScan(char *ip, unsigned long options, char *port_range)
{
	char buffer[80000];
	char cmd[500];
	FILE *pfp;
	int i;
	
	sprintf(cmd, "nmap %s -oX - -n", ip);

	switch(options & OS_SCAN_OPTION_SCAN_MASK)
	{
		case OS_SCAN_OPTION_TCP_CONNECT_SCAN:
			strcat(cmd, " -sT");
			break;
			
		case OS_SCAN_OPTION_TCP_SYN_SCAN:
			strcat(cmd, " -sS");
			break;
			
		case OS_SCAN_OPTION_STEALTH_FIN:
			strcat(cmd, " -sF");
			break;
			
		case OS_SCAN_OPTION_STEALTH_XMAS:
			strcat(cmd, " -sX");
			break;
			
		case OS_SCAN_OPTION_STEALTH_NULL:
			strcat(cmd, " -sN");
			break;
		
		default:
			break;
	}
	
	switch(options & OS_SCAN_OPTION_TIMIMG_MASK)
	{
		case OS_SCAN_OPTION_TIMIMG_PARANOID:
			strcat(cmd, " -T Paranoid");
			break;
		
		case OS_SCAN_OPTION_TIMIMG_SNEAKY:
			strcat(cmd, " -T Sneaky");
			break;
		
		case OS_SCAN_OPTION_TIMIMG_POLITE:
			strcat(cmd, " -T Polite");
			break;
		
		case OS_SCAN_OPTION_TIMIMG_NORMAL:
			strcat(cmd, " -T Normal");
			break;
		
		case OS_SCAN_OPTION_TIMIMG_AGGRESSIVE:
			strcat(cmd, " -T Aggressive");
			break;
		
		case OS_SCAN_OPTION_TIMIMG_INSANE:
			strcat(cmd, " -T Insane");
			break;
		
	}
	if(options & OS_SCAN_OPTION_OSSCAN)
		strcat(cmd, " -O");

	if(options & OS_SCAN_OPTION_UDP_SCAN)
		strcat(cmd, " -sU");

	if(options & OS_SCAN_OPTION_RPC_SCAN)
		strcat(cmd, " -sR");
	
	if(options & OS_SCAN_OPTION_IDENTD_SCAN)
		strcat(cmd, " -I");
	
	if(options & OS_SCAN_OPTION_FASTSCAN)
		strcat(cmd, " -F");
	
	if(options & OS_SCAN_OPTION_DONT_PING)
		strcat(cmd, " -P0");
	
	if(options & OS_SCAN_OPTION_USE_PORT_RANGE)
	{
		strcat(cmd, " -p ");
		strcat(cmd, port_range);
	}
	DEBUG(printf(__FUNCTION__ "(): scanning %s using '%s'\n", ip, cmd));
	pfp = popen(cmd, "r");
	if(pfp == NULL)
	{
		perror("I could not run nmap");
		return(NULL);
	}
	
	while(!feof(pfp))
	{
		if('<' == (buffer[0] = fgetc(pfp)))
		{
			break;
		}
	}
	i = 1;
	while(!feof(pfp) && i < sizeof(buffer))
	{
		buffer[i] = fgetc(pfp);
		if(feof(pfp))
			buffer[i] = '\0';
		i++;
	}
	pclose(pfp);
	if(i == sizeof(buffer))
	{
		fprintf(stderr, "hmm... it seems like nmap had a rather large output...\n");
		fprintf(stderr, "we ran out of buffer space for nmap's output when scanning %s\n", ip);
		buffer[sizeof(buffer) - 1] = '\0';
		fprintf(stderr, "dump of buffer=\n%s", buffer);
		return(NULL);
	}
	
	return(xmlParseDoc((xmlChar *)buffer));
}

nmap_scan_t *nmap_scan(char *ip, unsigned long options, char *ports)
{
	nmap_scan_t *ret = NULL;
	xmlDoc *doc;
	
	/* COMPAT: Do not genrate nodes for formatting spaces */
	LIBXML_TEST_VERSION xmlKeepBlanksDefault(0);

	doc = runNmapScan(ip, options, ports);
	if(doc == NULL)
	{
		fprintf(stderr, "cheops-agent: I could not run nmap... is it in your path?\n");
		return(NULL);
	}
	
	ret = parseNmapScan(doc);

	xmlCleanupParser();

	return(ret);
}
