/*
 * Cheops Next Generation GUI
 * 
 * cheops-gui.c
 * An agent GUI, for testing and communicating directly with agents
 *
 * Copyright(C) 1999 Brent Priddy <toopriddy@mailcity.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *                                                                  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <gnome.h>
#include "event.h"
#include "cheops-gui.h"
#include "gui-viewspace.h"
#include "gui-settings.h"
#include "gui-canvas.h"
#include "gui-help.h"
#include "gui-utils.h"
#include "logger.h"
#include "gui-handlers.h"
#include "gui-config.h"
#include "pixmaps/unknown.xpm"

//#define DEBUG_CHEOPS_GUI

#ifdef DEBUG_CHEOPS_GUI
	#define DEBUG(a) a
#else
	#define DEBUG(a)
#endif

/* duh... */
cheops_window *main_window;

/* should be the current notebook page */
net_page *current_page;

/* global preferences */
gui_preferences preferences;

/* agent to talk to the cheops-agent */
agent *primary_agent;
char *agent_ip = NULL;

/* this is for modal gnome_dialogs, so we can kill them if the upper level application is killed */
GtkWidget *kill_me_too = NULL;

int we_are_still_running = 1;

/* Buffer for creating all events, and event stuff, dont you love event driven programs :) */
char ebuf[65536];
event_hdr *eh = (event_hdr *)ebuf;
event *ee = (event *)(ebuf + sizeof(event_hdr));

static void do_save(GtkWidget *widget, int data);
void open_file_dialog(GtkWidget *w, gpointer p);
void make_save_dialog(char *title, char *filename, void *ok_clicked_function, void *cancel_clicked_function);
static void do_write_config_file_and_exit(GtkWidget *w, GtkWidget *fs);


/*
 * This is either called from opening a file while we have one alredy open or, we are quitting
 * and we should save the config
 */
void save_changes(int quit)
{
	int ask_again = TRUE;
	
	if(options_save_changes_on_exit && main_window->filename)
	{
		write_config_file(NULL,NULL);
		if(!quit)
		{
			open_file_dialog(NULL, NULL);

			return;
		}
	}
	else
	{
		if(main_window->filename == NULL)
		{
			switch(make_gnome_yes_no_dialog("Save Session?","Would you like to save your session?", FALSE, &ask_again))
			{
				case 0:
					if(!ask_again)
						options_save_changes_on_exit = TRUE;
					
					if(quit)
						make_save_dialog("Save As", NULL, do_write_config_file_and_exit, NULL);
					else
						make_save_dialog("Save As", NULL, open_file_dialog, NULL);
						
					return; // dont quit yet, open up a file save dialog
					
				case 1:
					if(!quit)
					{
						open_file_dialog(NULL, NULL);
						return;
					}
					break;
			}
		}
		else
		{
			switch(make_gnome_yes_no_dialog("Save Session?","Would you like to save your session?", FALSE, &ask_again))
			{
				case 0:
					if(!ask_again)
						options_save_changes_on_exit = TRUE;
					
					write_config_file(NULL, NULL);

					break;
				case 1:
					if(!quit)
					{
						open_file_dialog(NULL, NULL);
						return;
					}
					break;
			}
		}
	}
	gtk_main_quit();
}

void do_quit(GtkWidget *w, gpointer data)
{

	if(kill_me_too != NULL)
		gnome_dialog_close(GNOME_DIALOG(kill_me_too));

	we_are_still_running = FALSE;

	if(main_window->connected)
	{
		save_changes(TRUE);
	}
	else
		gtk_main_quit();
}

void quit(void)
{
	do_quit(NULL,NULL);
}

void destroy(GtkWidget *w, gpointer p)
{
	gtk_widget_destroy((GtkWidget *)p);
}

void showme(GtkWidget *widget, gpointer data)
{
	printf("event '%s'\n",(char *)data);	
} 

void do_page(GtkWidget *widget, gint data)
{
	GtkWidget *dialog = NULL;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *entry1;

	switch(data)
	{
		case 0:
			dialog = gnome_dialog_new("Add Viewspace",
							  GNOME_STOCK_BUTTON_OK,
							  GNOME_STOCK_BUTTON_CANCEL,
							  NULL);
			gnome_dialog_close_hides(GNOME_DIALOG(dialog), FALSE);
			
			entry1 = gnome_entry_new("new_viewspace");
			label = gtk_label_new("New page name:");
			hbox = gtk_hbox_new(FALSE, 5);
			gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(main_window->window));
			
			gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
			gtk_box_pack_start( GTK_BOX(hbox), entry1, FALSE, FALSE, 5);
			gtk_box_pack_start( GTK_BOX( GNOME_DIALOG(dialog)->vbox ), hbox, TRUE, TRUE, 0);

			gtk_widget_grab_focus(GTK_COMBO(entry1)->entry);

			gtk_signal_connect(GTK_OBJECT(GTK_COMBO(entry1)->entry),  "activate", GTK_SIGNAL_FUNC(click_ok_on_gnome_dialog), dialog);
			
			gtk_widget_show( hbox );
			gtk_widget_show( label );
			gtk_widget_show( entry1 );

			switch( gnome_dialog_run(GNOME_DIALOG(dialog)) )
			{
				case 0:
					add_viewspace(NULL, gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entry1)))) );
					gnome_dialog_close(GNOME_DIALOG(dialog));
					break;
				case 1:	
					gnome_dialog_close(GNOME_DIALOG(dialog));
					break;
				default:
					break;
			}
			break;
		case 1:
			if( make_gnome_yes_no_dialog("Remove Viewspace","Are you sure you want to delete this viewspace?",TRUE, NULL) == 0)
			{
				remove_viewspace(NULL, dialog);
			}
			break;
		default:
			break;
	}

}

static void do_network(GtkWidget *widget, gint data)
{
	GtkWidget *dialog;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *label2;
	GtkWidget *entry1;
	GtkWidget *entry2;
	char *c1, *c2;
	
	switch(data)
	{
		case 0:
		
			dialog = gnome_dialog_new("Add Host",
							  GNOME_STOCK_BUTTON_OK,
							  GNOME_STOCK_BUTTON_CANCEL,
							  NULL);
			gnome_dialog_close_hides(GNOME_DIALOG(dialog), FALSE);
			
			entry1 = gnome_entry_new("add_host_ip");
			label = gtk_label_new("Host / Address:");
			hbox = gtk_hbox_new(FALSE, 5);
			gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(main_window->window));
			
			gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
			gtk_box_pack_start( GTK_BOX(hbox), entry1, FALSE, FALSE, 5);
			gtk_box_pack_start( GTK_BOX( GNOME_DIALOG(dialog)->vbox ), hbox, TRUE, TRUE, 0);

			gtk_widget_grab_focus(GTK_COMBO(entry1)->entry);

			gtk_signal_connect(GTK_OBJECT(GTK_COMBO(entry1)->entry),  "activate", GTK_SIGNAL_FUNC(click_ok_on_gnome_dialog), dialog);
			
			gtk_widget_show( hbox );
			gtk_widget_show( label );
			gtk_widget_show( entry1 );

			switch( gnome_dialog_run(GNOME_DIALOG(dialog)) )
			{
				case 0:
					c1 = strdup(gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entry1)))) );
					
					add_network(NULL, c1, NULL); 
					if(c1)
						free(c1);
					
					gnome_dialog_close(GNOME_DIALOG(dialog));
					break;
				case 1:	
					gnome_dialog_close(GNOME_DIALOG(dialog));
					break;
				default:
					break;
			}
			break;

		case 1:
		
			dialog = gnome_dialog_new("Add Network",
							  GNOME_STOCK_BUTTON_OK,
							  GNOME_STOCK_BUTTON_CANCEL,
							  NULL);
			gnome_dialog_close_hides(GNOME_DIALOG(dialog), FALSE);
			
			entry1 = gnome_entry_new("add_host_ip");
			entry2 = gnome_entry_new("add_host_network");
			label = gtk_label_new("Network:");
			label2 = gtk_label_new("Netmask:");

			hbox = gtk_hbox_new(FALSE, 5);
			gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(main_window->window));
			
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			gtk_box_pack_start(GTK_BOX(hbox), entry1, FALSE, FALSE, 5);
			
			gtk_box_pack_start(GTK_BOX(hbox), label2, FALSE, FALSE, 5);
			gtk_box_pack_start(GTK_BOX(hbox), entry2, FALSE, FALSE, 5);

			gtk_box_pack_start( GTK_BOX( GNOME_DIALOG(dialog)->vbox ), hbox, TRUE, TRUE, 0);

			gtk_widget_grab_focus(GTK_COMBO(entry1)->entry);

			gtk_signal_connect(GTK_OBJECT(GTK_COMBO(entry2)->entry),  "activate", GTK_SIGNAL_FUNC(click_ok_on_gnome_dialog), dialog);
			
			gtk_widget_show( hbox );
			gtk_widget_show( label );
			gtk_widget_show( entry1 );
			gtk_widget_show( label2 );
			gtk_widget_show( entry2 );

			switch( gnome_dialog_run(GNOME_DIALOG(dialog)) )
			{
				case 0:
					c1 = strdup(gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entry1)))) );
					c2 = strdup(gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entry2)))) );
					
					add_network(NULL, c1, c2); 
					if(c1)
						free(c1);
					if(c2)
						free(c2);
					gnome_dialog_close(GNOME_DIALOG(dialog));
					break;
				case 1:	
					gnome_dialog_close(GNOME_DIALOG(dialog));
					break;
				default:
					break;
			}
			break;
		case 2:
		
			dialog = gnome_dialog_new("Add Network Range",
							  GNOME_STOCK_BUTTON_OK,
							  GNOME_STOCK_BUTTON_CANCEL,
							  NULL);
			gnome_dialog_close_hides(GNOME_DIALOG(dialog), FALSE);
			
			entry1 = gnome_entry_new("add_host_ip");
			entry2 = gnome_entry_new("add_host_ip");
			label = gtk_label_new("First IP:");
			label2 = gtk_label_new("Last IP:");

			hbox = gtk_hbox_new(FALSE, 5);
			gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(main_window->window));
			
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			gtk_box_pack_start(GTK_BOX(hbox), entry1, FALSE, FALSE, 5);
			
			gtk_box_pack_start(GTK_BOX(hbox), label2, FALSE, FALSE, 5);
			gtk_box_pack_start(GTK_BOX(hbox), entry2, FALSE, FALSE, 5);

			gtk_box_pack_start( GTK_BOX( GNOME_DIALOG(dialog)->vbox ), hbox, TRUE, TRUE, 0);

			gtk_widget_grab_focus(GTK_COMBO(entry1)->entry);

			gtk_signal_connect(GTK_OBJECT(GTK_COMBO(entry2)->entry),  "activate", GTK_SIGNAL_FUNC(click_ok_on_gnome_dialog), dialog);
			
			gtk_widget_show( hbox );
			gtk_widget_show( label );
			gtk_widget_show( entry1 );
			gtk_widget_show( label2 );
			gtk_widget_show( entry2 );

			switch( gnome_dialog_run(GNOME_DIALOG(dialog)) )
			{
				case 0:
					c1 = strdup(gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entry1)))) );
					c2 = strdup(gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entry2)))) );
					
					DEBUG(
						clog(LOG_DEBUG, "Discovering range %s to %s\n", c1, c2);
					);
					
					add_network_range(c1, c2); 
					if(c1)
						free(c1);
					if(c2)
						free(c2);
					gnome_dialog_close(GNOME_DIALOG(dialog));
					break;
				case 1:	
					gnome_dialog_close(GNOME_DIALOG(dialog));
					break;
				default:
					break;
			}
			break;
		default:
			break;
	}
}

static void do_map(GtkWidget *w, gpointer data)
{
	switch((int)data)
	{
		case 0:
			net_page_map_all(get_current_net_page());
			break;

		case 1:
			net_page_unmap_all(get_current_net_page());
			net_page_map_all(get_current_net_page());
			break;
			
		default:
			printf("do_map(): there should not be a default\n");
	}
}

static void do_unmap(GtkWidget *w, gpointer data)
{
	net_page_unmap_all(get_current_net_page());
}

static void do_select_all(GtkWidget *w, gpointer data)
{
	net_page_highlight_all(get_current_net_page(), TRUE);
	net_page_set_plink();
}

/*
 * Callback for the menu delete selection (it creates a confirm dialog)
 */
void do_delete_page_object(GtkWidget *w, gpointer data)
{
	int ask_again = TRUE;
/* check and see if there is nothing selected */
	if((get_current_net_page())->plink == NULL)
		return;

	if(options_confirm_delete)
	{
		if(make_gnome_yes_no_dialog("Delete?!?","Are you Sure you want to delete all Highlighted items?",TRUE, &ask_again) == 0 )
		{
			net_page_delete_page_object(get_current_net_page());
			if(!ask_again)
				options_confirm_delete = FALSE;
		}
	}
	else
		net_page_delete_page_object(get_current_net_page());
	
}

static void do_write_config_file_and_exit(GtkWidget *w, GtkWidget *fs)
{
	write_config_file(w,fs);
	gtk_main_quit();
}


void make_save_dialog(char *title, char *filename, void *ok_clicked_function, void *cancel_clicked_function)
{
	GtkWidget *fs;
	
	if((fs = gtk_file_selection_new(title)) != NULL)
	{
		gtk_window_set_position(GTK_WINDOW(fs), GTK_WIN_POS_CENTER);
		if(filename)
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), filename);
		else
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs),CHEOPS_DEFAULT_FILENAME);

		if(ok_clicked_function)
			gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", ok_clicked_function, fs);
		else
			gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", write_config_file, fs);
			
		if(cancel_clicked_function)
			gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button), "clicked", ok_clicked_function, fs);
		else
			gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button), "clicked", kill_widget_argument, fs);

		gtk_signal_connect(GTK_OBJECT(fs), "delete_event", gtk_widget_destroy, fs);
		gtk_widget_show_all(fs);
	}
}

static void do_save(GtkWidget *widget, int data)
{
	if((main_window->filename != NULL) && !data)
	{
		write_config_file(NULL,NULL); /* this will write to the existing filename */
		return;
	}
	else
		make_save_dialog("Save As", main_window->filename, (void *)write_config_file, NULL);

}

void open_file_and_read_config_file(GtkWidget *w, GtkWidget *fs)
{
	net_page_remove_all();
	read_config_file(w, fs);
}

void open_file_dialog(GtkWidget *w, gpointer p)
{
	GtkWidget *fs = gtk_file_selection_new("Open");
	if(fs)
	{
		gtk_window_set_position(GTK_WINDOW(fs), GTK_WIN_POS_CENTER);
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs),CHEOPS_DEFAULT_FILENAME);
		gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", open_file_and_read_config_file, fs);
		gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button), "clicked", kill_widget_argument, fs);
		gtk_signal_connect(GTK_OBJECT(fs), "delete_event", gtk_widget_destroy, fs);
		gtk_widget_show_all(fs);
	}
}

static void do_open(GtkWidget *widget, gpointer data)
{
	save_changes(FALSE); // don't quit	
}

int start_ipv4_agent(char *hostname)
{
	if ((primary_agent = event_request_agent(AGENT_TYPE_IPV4, hostname, 1)))
	{
		agent_ip = makestring(hostname);
		printf("Connected to ipv4 agent.\n\n");
		register_gui_handlers();
		return(1);
	} 
	else 
	{
		clog(LOG_ERROR, "Unable to connect to local agent\n");
		return(0);
	}
}

void connect_to_ipv4_agent(GtkWidget *w, char *host)
{
	if (strlen(host))
	{
		if((main_window->connected = start_ipv4_agent(host)))
		{
			if(main_window->agent_ip)
				free(main_window->agent_ip);
				
			main_window->agent_ip = makestring(host);
		}
	}

	if(!main_window->connected)
	{
		make_gnome_ok_dialog("Could not connect to agent", "I could not connect to the Specified IP agent");
	}
}

void make_main_window(cheops_window *w, int connected)
{
	static GtkItemFactoryEntry menu_items[] = 
	{
		{ "/_File",                             NULL,         NULL,                  0, "<Branch>" },
		{ "/File/tearoff1",                     NULL,         NULL,                  0, "<Tearoff>" },
		{ "/File/_Open...",                     "<control>O", do_open,               0 },
		{ "/File/_Save",                        "<control>S", do_save,               0 },
		{ "/File/Save _As",                     "",           do_save,               1 },
		{ "/File/sep1",                         NULL,         NULL,                  0, "<Separator>" },
		{ "/File/_Quit",                        "<control>Q", do_quit,               0 },
		{ "/_Edit",                             NULL,         NULL,                  0, "<Branch>" },
		{ "/Edit/tearoff1",                     NULL,         NULL,                  0, "<Tearoff>" },
		{ "/Edit/S_ettings...",                 "<control>E", do_settings,           0 },
		{ "/Edit/sep2",                         NULL,         NULL,                  0, "<Separator>" },
		{ "/Edit/Select _All...",               "<control>A", do_select_all,         0 },
		{ "/Edit/_Delete host...",              "<control>D", do_delete_page_object, 0 },
		{ "/_Viewspace",                        NULL,         NULL,                  0, "<Branch>" },
		{ "/Viewspace/tearoff1",                NULL,         NULL,                  0, "<Tearoff>" },
		{ "/Viewspace/_New...",                 "<control>N", do_page,               0 },
		{ "/Viewspace/_Remove",                 NULL,         do_page,               1 },
		{ "/Viewspace/sep1",                    NULL,         NULL,                  0, "<Separator>" },
		{ "/Viewspace/Add _Host...",            "<control>H", do_network,            0 },
		{ "/Viewspace/Add N_etwork...",         "<control>N", do_network,            1 },
		{ "/Viewspace/Add Network R_ange...",   "<control>R", do_network,            2 },
		{ "/Viewspace/sep2",                    NULL,         NULL,                  0, "<Separator>" },
		{ "/Viewspace/Map",                     NULL,         NULL,                  0, "<Branch>" },
		{ "/Viewspace/Map/_Map Everything",     "<control>M", do_map,                0 },
		{ "/Viewspace/Map/ReMap Everything",    NULL,         do_map,                1 },
		{ "/Viewspace/Map/UnMap Everything",    "<control>U", do_unmap,              0 },
		{ "/_Help",                             NULL,         NULL,                  0, "<LastBranch>" },
		{ "/Help/About...",                     NULL,         do_about,              0 },
		{ "/Help/License...",                   NULL,         do_license,            0 },
	};
	
	int nmenu_items = sizeof(menu_items)/sizeof(menu_items[0]);
	char buf[256];
	FILE *fp;
	GtkWidget *dialog;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *entry1;

	w->net_pages = NULL;
	w->filename = NULL;
	w->connected = connected;
	
	w->window = gnome_app_new(CHEOPS_TITLE, CHEOPS_TITLE);
	gtk_widget_set_usize(w->window, CHEOPS_MIN_WIDTH, CHEOPS_MIN_HEIGHT);
	gtk_signal_connect(GTK_OBJECT(w->window),"delete-event", GTK_SIGNAL_FUNC(do_quit),NULL);
	gtk_window_set_position(GTK_WINDOW(w->window), GTK_WIN_POS_CENTER);
		
	w->accel_group = gtk_accel_group_new();
	
	w->itemf = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<blah>",w->accel_group);
	gtk_item_factory_create_items(w->itemf, nmenu_items, menu_items, NULL);
	w->menu = gtk_item_factory_get_widget(w->itemf, "<blah>");
	
	gtk_accel_group_attach(w->accel_group, GTK_OBJECT(w->window));
	
	w->vbox = gtk_vbox_new(FALSE,0);
	w->notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->notebook), GTK_POS_TOP);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(w->notebook), TRUE);

	gnome_app_set_contents(GNOME_APP(w->window), w->vbox);
	gtk_box_pack_start(GTK_BOX(w->vbox), w->menu, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(w->vbox), w->notebook, TRUE, TRUE, 0);

	gtk_widget_show(w->notebook);
	gtk_widget_show(w->vbox);
	gtk_widget_show(w->menu);
	gtk_widget_show(w->window);

//	while(gtk_events_pending())
//		gtk_main_iteration();
		
	main_window->agent_ip = agent_ip;
	
	init_net_page_stuff();
		
/* if we did not connect, that means that we could not connect to a local agent, so try a ipv4 one */
	while(!w->connected)
	{
		/* do something to give them a prompt to loginto a netaork agent */
		dialog = gnome_dialog_new("Connect to Agent",
						  GNOME_STOCK_BUTTON_OK,
						  GNOME_STOCK_BUTTON_CANCEL,
						  NULL);
		gnome_dialog_close_hides(GNOME_DIALOG(dialog), FALSE);
		
		entry1 = gnome_entry_new("");
		label = gtk_label_new("Agent hostname:");
		hbox = gtk_hbox_new(FALSE, 5);
		gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(main_window->window));
		
		gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
		gtk_box_pack_start( GTK_BOX(hbox), entry1, FALSE, FALSE, 5);
		gtk_box_pack_start( GTK_BOX( GNOME_DIALOG(dialog)->vbox ), hbox, TRUE, TRUE, 0);

		gtk_widget_grab_focus(GTK_COMBO(entry1)->entry);

		gtk_signal_connect(GTK_OBJECT(GTK_COMBO(entry1)->entry),  "activate", GTK_SIGNAL_FUNC(click_ok_on_gnome_dialog), dialog);
			
		gtk_widget_show( hbox );
		gtk_widget_show( label );
		gtk_widget_show( entry1 );

		switch( gnome_dialog_run(GNOME_DIALOG(dialog)) )
		{
			case 0:
				connect_to_ipv4_agent(NULL, gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entry1)))));
				gnome_dialog_close(GNOME_DIALOG(dialog));
				break;
			case 1:	
				gnome_dialog_close(GNOME_DIALOG(dialog));
				exit(0);
			default:
				break;
		}
	}
	
/* lets open the default ~/.gnome/cheops-ng.map file to read the config */
	sprintf(buf,"%s/%s",getenv("HOME"), CHEOPS_DEFAULT_FILENAME_NOT_HOME);

	if(NULL == (fp = fopen(buf,"r")))
	{
		w->filename = NULL;
		make_viewspace("My Network Stuff");
	}
	else
	{
		fclose(fp);
		w->filename = makestring(buf);
	}
	read_config_file(NULL,NULL);

}

int main(int argc, char *argv[])
{
	int connected=0;
	char *ip;

	gnome_init(CHEOPS_TITLE, CHEOPS_VERSION ,argc,argv);

	if (!getenv("HOME"))
	{
		fprintf(stderr, "No home directory!\n");
		return(1);
	}
	else
	{
		if((primary_agent = event_request_agent(AGENT_TYPE_LOCAL, NULL, 1)))
		{	
			printf("Connected to local agent.\n\n");
			register_gui_handlers();
			connected = 1;
		}
		else 
		{
			connected = 0;

			if((ip = config_get_agent_ip_address()) != NULL)
			{
				connected = start_ipv4_agent(ip);
				free(ip);
			}
		}
	}

	if( (main_window = NEW(cheops_window)) == NULL)
	{
		quit();
	}
	make_main_window(main_window, connected);

	gnome_dns_init(2); //use 2 dns servers
	gtk_main();
	return(0);
}

