/*
 * Cheops Next Generation GUI
 * 
 * agent-osscan.c
 * Operating system discovery code
 *
 * Copyright(C) 1999 Brent Priddy <toopriddy@mailcity.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *                                                                  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA
 */
#include <sys/time.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>
#include <glib.h>
#include <sched.h>
#include <semaphore.h>
#include "event.h"
#include "io.h"
#include "sched.h"
#include "cheops-sh.h"
#include "logger.h"
#include "cheops-agent.h"
#include "agent-osscan.h"
#include "ip_utils.h"
#include "cache.h"

//#define DEBUG_OS_SCAN

#ifdef DEBUG_OS_SCAN
	#define DEBUG(a) a
#else
	#define DEBUG(a)
#endif

pthread_mutex_t os_list_mutex;
pthread_mutex_t os_read_mutex;
pthread_mutex_t os_write_mutex;
pthread_t os_detect_thread;
sem_t os_detect_go_sem;

typedef struct _os_detect_queue {
	char *name;
	void *np;
	void *agent;
	unsigned int flags;
	char ports[256];
} os_detect_queue;

GList *os_detect_q = NULL;
int os_detect_q_outstanding = 0;

void *dequeue_osscan(void *arg);
int do_osscan (char *name, void *np, void *agent);
void enqueue_osscan(char *name, void *np, void *agent, char *ports, unsigned int flags);
int osscan_timer(void *data);

event_hdr *event_to_send = NULL;
agent *agent_to_send = NULL;

char *scan_ports;
unsigned int scan_flags;

int handle_osscan_request(event_hdr *h, event *e, agent *a)
{
	char	*name = ip2str(ntohl(e->os_scan_e.ip));

	DEBUG(printf("%s(): enqueueing scan\n", __FUNCTION__));	
	enqueue_osscan(name,
	               e->os_scan_e.np ,
	               a,
	               e->os_scan_e.ports,
	               ntohl(e->os_scan_e.options));
	return(0);
}

void init_osscan(void)
{
	DEBUG(printf("%s(): Initalizing the muteciessesss\n", __FUNCTION__));
	pthread_mutex_init(&os_list_mutex, NULL);
	pthread_mutex_init(&os_read_mutex, NULL);
	pthread_mutex_init(&os_write_mutex, NULL);
	
	pthread_mutex_lock(&os_read_mutex);
	
	sem_init(&os_detect_go_sem, 0, 0);
	DEBUG(printf("%s(): finished Initalizing the muteciessesss\n", __FUNCTION__));
	
	cheops_sched_add(800, osscan_timer, NULL);
		
	pthread_create(&os_detect_thread, NULL, dequeue_osscan, NULL);
}

void enqueue_osscan(char *name, void *np, void *agent, char *ports, unsigned int flags)
{
	os_detect_queue *q;

	DEBUG(printf("%s(): enqueueing %s\n", __FUNCTION__, name));

	if(name && np && agent)
	{
		q = malloc(sizeof(os_detect_queue));
		if(!q)
			return;
		
		q->name = strdup(name);
		q->np = np;
		q->agent = agent;
		q->flags = flags;
		strncpy(q->ports,ports,sizeof(q->ports));
		
		pthread_mutex_lock(&os_list_mutex);
		os_detect_q_outstanding++;

		DEBUG(printf("%s(): enqueueing %s aquired list mutex\n", __FUNCTION__, name));
		DEBUG(printf("%s(): %d outstanding entries\n", __FUNCTION__, os_detect_q_outstanding));
		os_detect_q = g_list_append(os_detect_q, q);
		
		pthread_mutex_unlock(&os_list_mutex);

		sem_post(&os_detect_go_sem);
	}
}

void *dequeue_osscan(void *arg)
{
	os_detect_queue *q = NULL;
	GList *gl;
	
	while(1)
	{
		DEBUG(printf("%s(): dequeue going to sleep\n", __FUNCTION__));
		sem_wait(&os_detect_go_sem);
		DEBUG(printf("%s(): dequeue woken up\n", __FUNCTION__));

		pthread_mutex_lock(&os_list_mutex);
		DEBUG(printf("%s(): dequeue aquired mutex \n", __FUNCTION__));
		if( (gl = g_list_first(os_detect_q)) )
		{
			if( (q = gl->data) )
			{
				os_detect_q_outstanding--;
				DEBUG(printf("%s(): %d outstanding entries\n", __FUNCTION__, os_detect_q_outstanding));
				os_detect_q = g_list_remove(os_detect_q, q);
			}
			else
			{
				DEBUG(printf("%s(): dequeue'ed a NULL data pointer\n", __FUNCTION__));
			}
		}
		else
		{
			DEBUG(printf("%s(): dequeue did not dequeue anything\n", __FUNCTION__));
		}
		DEBUG(printf("%s(): dequeue released mutex \n", __FUNCTION__));
		pthread_mutex_unlock(&os_list_mutex);
		
		if(gl == NULL || q == NULL)
		{
			continue;
		}
		else
		{
			DEBUG(printf("%s(): starting os scan on %s\n", __FUNCTION__, q->name));
			scan_ports = q->ports;
			scan_flags = q->flags;
			do_osscan(q->name, q->np, q->agent);
				
			DEBUG(printf("%s(): finished os scan on %s\n", __FUNCTION__, q->name));

			if(q->name)
				free(q->name);
			if(q)
				free(q);
		}
	}
	printf("%s(): I returned?!?!?\b\n", __FUNCTION__);
	return(NULL);
}

void agent_osscan_send_event(agent *a, event_hdr *e)
{
	event_to_send = e;
	agent_to_send = a;
	pthread_mutex_unlock(&os_read_mutex);
}

int osscan_timer(void *data)
{
	if( EBUSY != pthread_mutex_trylock(&os_read_mutex) )
	{
		if (event_send (agent_to_send, event_to_send) < 0)
		{
			clog (LOG_WARNING, "Unable to send osscan reply\n");
			return (0);
		}
		pthread_mutex_unlock(&os_write_mutex);
	}
	return(1); // keep it running
}
