/*
 * Copyright (c) 1998, 1999
 * Stefan Savage and the University of Washington.
 * All rights reserved. 
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author(s) may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.  
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR(S) ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR(S) BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * HISTORY
 * 04-Nov-99  Stefan Savage (savage) at the University of Washington
 *	Port to Linux
 *
 * 03-Nov-99  Stefan Savage (savage) at the University of Washington
 *	Cleanup, removed overlapping big packets (bug)
 *
 * 01-Apr-99  Stefan Savage (savage) at the University of Washington
 *	Real request strings and limited receiver window.  AD tests.
 *
 * 21-Mar-99  Stefan Savage (savage) at the University of Washington
 *	Switched to OOO ack parity, overlapping big packets
 *
 * 21-Dec-98  Stefan Savage (savage) at the University of Washington
 *	Created.
 */

#include <stdio.h>
#include <math.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h> 
#ifdef __FreeBSD__
#include <netinet/ip_fw.h>
#endif /* __FreeBSD__ */
#ifdef linux
struct tcphdr {char x[20];};
struct iphdr {char x[20];};
struct udphdr {char x[20];};
struct icmphdr {char x[20];};
#include <linux/if.h>
#include <linux/ip_fw.h> 
#endif /* linux */
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include "inet.h"
#include "capture.h"
#include "setsignal.h"
#include "distribution.h"

#define DISTRIBUTION_EXPONENTIAL	(0)
#define DISTRIBUTION_PERIODIC		(1)
#define DISTRIBUTION_UNIFORM		(2)
char *distributionNames[] = {"exponential","periodic","uniform"};

#define DEFAULT_DISTRIBUTION	DISTRIBUTION_UNIFORM
#define DEFAULT_MEAN		(0.1)
#define DEFAULT_COUNT		(100)
#define DEFAULT_TARGETPORT	(80)
#define DEFAULT_PACKETSIZE	(41)

char defaultRequest[]="GET / HTTP/1.0\nAccept: text/plain\nAccept: */*\nUser-Agent: Mozilla/4.0 (compatible; MSIE 5.0; Windows NT; DigExt; Sting)\n\n";

#define SYNTIMEOUT		(3.0)	/* 3 seconds */
#define MAXSYNRETRANSMITS	(3)
#define MAXDATARETRANSMITS	(5)
#define INTERPHASEDELAY		(2.0)	/* 2 seconds after data seeding */
#define SEQSKIP			(3)	/* size of "fake" hole */

char *progName;

extern char *optarg;	/* Variables used for getopt(3) */
extern int optind;
extern int opterr;
extern int optopt;

struct TcpSession session;
int acksReceived = 0;
int dataSent = 0;
int dataSeenSent = 0;
int dataReceived = 0;
int acksSent = 0;
int dataLost = 0;
int acksLost = 0;

int verbose = 0;
int test = 0;
int initSession = 0;
int initCapture = 0;
int initFirewall = 0;
int killOldFlows = 1;

#ifdef __FreeBSD__
struct ip_fw firewallRule;
#endif /* __FreeBSD__ */
#ifdef linux
struct ip_fwchange firewallRule;
#endif /* linux */

void SendReset(struct TcpSession *s)
{
  struct TcpPacket p;
  int i;

  for (i=0;i<3;i++) {  /* 3 is totally arbitrary */
    SendSessionPacket(s, &p, sizeof(struct TcpPacket), TCPFLAGS_RST);
  }
}

void RespondWithReset(struct TcpPacket *p) 
{
  struct TcpSession s;

  memset(&s, 0, sizeof(struct TcpSession));
  printf("Resetting connection\n");
  s.src = p->ip.ip_dst;
  s.dst = p->ip.ip_src;
  s.dport = ntohs(p->tcp.tcp_sport);
  s.sport = ntohs(p->tcp.tcp_dport);
  s.snd_nxt = ntohl(p->tcp.tcp_ack);
  s.rcv_nxt = ntohl(p->tcp.tcp_seq);
  SendReset(&s);
}


/* make a clean exit on interrupts */
RETSIGTYPE
Cleanup(int signo)
{


  /* If a firewall rule has been installed then remove it */
  if (initFirewall > 0) {
#ifdef linux
#define IP_FW_DEL	(IP_FW_DELETE)
#endif /* linux */
    if (setsockopt(session.socket,IPPROTO_IP, IP_FW_DEL,
		   &firewallRule, sizeof(firewallRule)) != 0) {
      printf("ERROR: couldn't remove firewall rule\n");
    }
  }

  if (initSession > 0) {
    SendReset(&session);
    shutdown(session.socket,2);
  }
  
  if (initCapture > 0) {
    CaptureEnd();
  }
  exit(0);
}

void Quit()
{
  fflush(stdout);
  fflush(stderr);
  Cleanup(1);
  exit(-1);
}

void Usage() 
{
  printf("usage:\n%s:",progName);
  printf("\t [-p target-port] [-c count] [-s packet-size]\n");
  printf("\t [-f {exponential|uniform|periodic}] [-m mean]\n");
  printf("\t [-w source-port] [-h request-file] [-v]\n");
  Quit();
}

int BindTcpPort(int sockfd)
{
   struct sockaddr_in  sockName;
   int port, result;
   int randomOffset;

#define START_PORT (10*1024) 
#define END_PORT   (0xFFFF)

   /* Choose random offset to reduce likelihood of collision with last run */
   randomOffset = (int)(1000.0*drand48());

   /* Try to find a free port in the range START_PORT+1..END_PORT */
   port = START_PORT+randomOffset;
   do {
       ++port;
       sockName.sin_addr.s_addr = INADDR_ANY;       
       sockName.sin_family = AF_INET;
       sockName.sin_port = htons(port);
       result = bind(sockfd, (struct sockaddr *)&sockName,
                     sizeof(sockName));
   } while ((result < 0) && (port < END_PORT));
   
   if (result < 0) {
     /* No free ports */
       perror("bind");
       port = 0;
   }   
   return port;
}


void PrintTimeStamp(struct timeval *ts)
{
	(void)printf("%02d:%02d:%02d.%06u ",
		     (unsigned int)ts->tv_sec / 3600,
		     ((unsigned int)ts->tv_sec % 3600) / 60,
		     (unsigned int)ts->tv_sec % 60, (unsigned int)ts->tv_usec);
}

double GetTime()
{
  struct timeval tv;
  struct timezone tz;
  double postEpochSecs;

  if (gettimeofday(&tv, &tz) < 0) {
    perror("GetTime");
    exit(-1);
  }
  
  postEpochSecs = (double)tv.tv_sec + ((double)tv.tv_usec/(double)1000000.0);
  return postEpochSecs;
}

/*
 * Does a packet belong to a directional flow in our session?
 */
#define INSESSION(p, src, sport, dst, dport)			\
    (((p)->ip.ip_src == (src)) && ((p)->ip.ip_dst == (dst)) &&	\
     ((p)->ip.ip_p == IPPROTOCOL_TCP) &&			\
     ((p)->tcp.tcp_sport == htons(sport)) &&			\
     ((p)->tcp.tcp_dport == htons(dport)))


/* XXX
 * TODO:
 * fix packet sizing
 * check receiver window, mss
 * Write in-order byte every x seconds
 */
void DataSeeding(struct TcpSession *s, char *data, int packetSize,
		 int count, double *offsets)
{

  struct TcpPacket *pkt;
  struct TcpPacket *p;
  struct PacketInfo pi;
  double startTime,currentTime, finalTime;
  uint32 lastSeqSent;
  char *dataPtr;
  int dataOffset;
  int dataSize = packetSize-sizeof(struct TcpPacket);
  double *wireOffsets;
  int done=0;
  int i;


  /* Convert from interval format to offsets from 0  */
  offsets[0] = 0;
  for (i=0;i<count-1;i++) {
    offsets[i+1] = offsets[i] + offsets[i+1];
  }
  
  /* Allocate space for wiretime measurements */
  if ((wireOffsets = (double *)malloc(count*sizeof(double))) == NULL) {
    printf("ERROR: out of memory\n");
    Quit();
  }    

  /* Allocate space for packet and make pointer to data area */
  if ((pkt = (struct TcpPacket *)malloc(packetSize)) == NULL) {
    printf("ERROR: out of memory\n");
    Quit();
  }
  dataPtr = (char *)pkt+sizeof(struct TcpPacket);

  lastSeqSent = s->snd_nxt; /* right window edge */
  s->snd_nxt += SEQSKIP;  /* Ensure that data stream is OOO */
  startTime = GetTime();

  /*
   * Main loop.  The termination conditions are quite ugly.  In the begining,
   * each iteration checks if a packet should be sent and otherwise checks
   * for received packets.  When the last packet has been sent, dataSent == n,
   * then the finalTime variable is set.  This gets checked in the second
   * branch of the first conditional (only called when dataSent >= n).
   * During the remaining time, we just check for incoming packets.  
   * The timout must be large enough that we expect to see all the remaining 
   * packets before out time is up.
   * When we run out of time the done flag is set.
   */
  while(!done) {
    currentTime = GetTime();
    /* Are there packets left to sent? */
    if (dataSent < count) {
      /* Is it time to send a new packet? */
      if (currentTime >= startTime + offsets[dataSent]) {
	/* Yup, fill in the packet and sent it */
	dataOffset = s->snd_nxt - s->iss - 1;
	memcpy((void *)dataPtr,( void *)&(data[dataOffset]),dataSize);
	SendSessionPacket(s, pkt, packetSize, TCPFLAGS_PSH | TCPFLAGS_ACK);
	s->snd_nxt += dataSize; 
	/*
	 * XXX
	 * Alternative "skip-style" overlapping algorithm
	 * Needs complementary code in hole filing
	 * Reduces buffer requirement to (count*dataSize+1)/2 
	 *
	 * if (dataSize > 1) {
	 *   if ((dataSent % 2) == 0) {
	 *     s->snd_nxt += dataSize;
	 *   }
	 * }
	 */
	dataSent++;
	/* If this is the last packet then set the final timeout */
	if (dataSent == count) {
	  finalTime = currentTime+INTERPHASEDELAY;
	}
      }
    } else {
      /* Ok, we've waited for any stragler packets */
      if (currentTime >= finalTime) {
	done = 1;
      }
    }

    /* Now check if we have received any packets */
    if ((p =(struct TcpPacket *)CaptureGetPacket(&pi)) != NULL) {
      /* Do we see the data packet we just sent? */
      if (INSESSION(p,s->src,s->sport,s->dst,s->dport) &&
	   (p->tcp.tcp_flags == (TCPFLAGS_PSH | TCPFLAGS_ACK)) &&
	   (ntohl(p->tcp.tcp_seq) > lastSeqSent) &&
	   (ntohl(p->tcp.tcp_ack) == s->rcv_nxt)) {
	/* Check wiretime at which it was sent */
	wireOffsets[dataSeenSent] =
	  (double) pi.ts.tv_sec+ (pi.ts.tv_usec/1000000.0);
	dataSeenSent++;
	lastSeqSent = ntohl(p->tcp.tcp_seq);  /* advance right window edge */
	if (verbose) {
	  putchar('_');
	  fflush(stdout);
	}       
      } /* Do we see an expected ack in return? */
      else if (INSESSION(p,s->dst,s->dport,s->src,s->sport) &&
	       (p->tcp.tcp_flags & TCPFLAGS_ACK) &&
	       (ntohl(p->tcp.tcp_seq) == s->rcv_nxt) &&
	       (ntohl(p->tcp.tcp_ack) == s->snd_una)) {
	acksReceived++;
	/* Has the ttl changed? */
	if (p->ip.ip_ttl != s->ttl) {
	  printf("WARNING: route may have changed (ttl was %d, is  %d).\n",
		 s->ttl, p->ip.ip_ttl);
	  s->ttl = p->ip.ip_ttl;
	}
	if (verbose) {
	  putchar('-');
	  fflush(stdout);
	}
      }
      /* From them to us and a reset... target doesn't like us */
      else if (INSESSION(p,s->dst,s->dport,s->src,s->sport) &&
	       (p->tcp.tcp_flags & TCPFLAGS_RST)) {
	printf("WARNING: Early Reset during data seeding\n");
	Quit();
      }
      /* Is it some other packet between us? */
      else if (INSESSION(p,s->src,s->sport,s->dst,s->dport) ||
	       INSESSION(p,s->dst,s->dport,s->src,s->sport)) {
	/* Unknown packet, stop the presses */
	printf("ERROR: Unexpected packet received during data seeding\n");
	printf("Expecting:\n");
	printf("\tsrc = %s:%d (seq>%u, ack=%u)\n",
	       InetAddress(s->src), s->sport,
	       lastSeqSent, s->rcv_nxt);
	printf("\tdst = %s:%d (seq=%u, ack=%u)\n",
	       InetAddress(s->dst),s->dport,
		 s->rcv_nxt, s->snd_una);
	printf("Received:\n\t");
	PrintTimeStamp(&pi.ts);
	PrintTcpPacket(p);
	Quit();
      }
      /* Its from some other flow */
      else {
	printf("WARNING: Received packet from different flow\n");
	PrintTimeStamp(&pi.ts);
	PrintTcpPacket(p);
	if (killOldFlows) {
	  if (p->ip.ip_dst == s->src && !(p->tcp.tcp_flags & TCPFLAGS_RST)) {
	    RespondWithReset(p);
	  }
	}
      }
    }
  }

  /*
   * Calculate actual wire-time intervals and return them via overloaded
   * offsets array.
   */
  for (i=0;i<count-1;i++) {
    offsets[i] = wireOffsets[i+1] - wireOffsets[i];
  }

}


void HoleFilling(struct TcpSession *s, char *data, int packetSize)
{
  struct TcpPacket *packet;
  struct TcpPacket *p;
  struct PacketInfo pi;
  uint32 lastSeqSent = s->snd_nxt;  /* right window edge */
  int numRetransmits = 0;
  double timeoutTime;
  char *dataPtr;
  int dataOffset;
  int timeoutDelay = 2.0;  /* 2 seconds */
  int dataSize = dataSize = SEQSKIP;  /* size of initial "hole" to be filled */
  
  if ((packet = (struct TcpPacket *)malloc(packetSize)) == NULL) {
    printf("ERROR: Out of memory\n");
    Quit();
  }
  dataPtr = (char *)packet+sizeof(struct TcpPacket);
  s->snd_nxt = s->snd_una;

  /* while there is still un-acked data to deliver */
  while(s->snd_nxt < lastSeqSent && numRetransmits < MAXDATARETRANSMITS) {
    /*
     * Fill packet with "consistent" data in sequence space
     * Important for servers that do abortive close when parse fails
     */
    dataOffset = s->snd_nxt - s->iss - 1;
    memcpy((void *)dataPtr,(void *)&data[dataOffset],dataSize);

    /* Retransmit packet */
    SendSessionPacket(s, packet, sizeof(struct TcpPacket)+dataSize,
		      TCPFLAGS_PSH | TCPFLAGS_ACK);
    numRetransmits++;
    
    timeoutTime=GetTime()+timeoutDelay;

    
    /* Wait for an ACK from the target */
    while(GetTime() < timeoutTime) {
      /* Got a packet */
      if ((p =(struct TcpPacket *)CaptureGetPacket(&pi)) != NULL) {
	/* From us to them? Ignore*/
	if (INSESSION(p,s->src,s->sport,s->dst,s->dport)) {
	} 
	/* From them to us and an ACK */
	else if (INSESSION(p,s->dst,s->dport,s->src,s->sport) &&
		   (p->tcp.tcp_flags & TCPFLAGS_ACK) &&
		   (ntohl(p->tcp.tcp_seq) == s->rcv_nxt)) {
	  /* Does it cover the hole? */
	  if (ntohl(p->tcp.tcp_ack) > s->snd_nxt) {
	    /*
	     * Yup, rocord that data was lost, move snd_una to next hole
	     * and move on.
             */
	    dataLost++;
	    numRetransmits = 0;
	    /* XXX, This is ugly... it undoes the SEQSKIP value */
	    dataSize = packetSize - sizeof(struct TcpPacket);
	    s->snd_una = ntohl(p->tcp.tcp_ack);
	    s->snd_nxt = s->snd_una;
	    timeoutTime = GetTime();
	    if (verbose && dataLost > 1) { /* don't print first retransmit */
	      putchar('*');
	      fflush(stdout);
	    }
	  }
	}
	/* From them to us and a reset... target doesn't like us */
	else if (INSESSION(p,s->dst,s->dport,s->src,s->sport) &&
		   (p->tcp.tcp_flags &TCPFLAGS_RST)) {
	  printf("\nWARNING: Early Reset during hole filling\n");
	  timeoutTime = GetTime(); 
	  numRetransmits = MAXDATARETRANSMITS;
	}
	/* otherwise, its something weird */
	else {
	  printf("\nWARNING: Received unexpected packet during hole filling\n");
	  PrintTimeStamp(&pi.ts);
	  PrintTcpPacket(p);
	  if (killOldFlows) {
	    if (p->ip.ip_dst == s->src && !(p->tcp.tcp_flags & TCPFLAGS_RST)) {
	      RespondWithReset(p);
	    }
	  }
	}
      }
    }
  }

  printf("\n");

  /* Subtract one from dataLost to account for artificial hole in begining */
  dataLost--;  

  /*
   * If we exited without filling all holes then alert the user and
   * tell them what we DO know.
   */
  if (numRetransmits == MAXDATARETRANSMITS) {
    printf("WARNING: Could not fill hole.  Results may be incorrect.\n");
    printf("Last filled byte = %d\n", dataOffset);
    printf("Last acked packet was %d\n", (dataOffset-SEQSKIP)/dataSize);
    if (dataSent-dataLost-acksReceived > 0) {
      printf("WARNING: Assuming that %d additional data packets were lost\n",
	     (dataSent-acksReceived) - dataLost);
    }
    dataLost += (dataSent-acksReceived) - dataLost;
  }

  /* Sanity: make sure that we observed every packet that we sent */
  if (dataSeenSent != dataSent) {
    printf("WARNING: Didn't notice %d packets sent\n", dataSent-dataSeenSent);
  }
}

/*
 * Setup a TCP session between our machine and the specified target
 * Extract and return the key bits of TCP state
 */
int EstablishSession(uint32 sourceAddress, uint16 sourcePort,
		     uint32 targetAddress, uint16 targetPort,
		     struct TcpSession *s)
{
  int rawSocket;
  struct TcpPacket *p;
  struct TcpPacket synPacket;
  struct PacketInfo pi;
  int synAckReceived = 0;
  int numRetransmits = 0;
  double timeoutTime;
  double ts1, ts2;
  int flag=1;

  /* Initialize session structure */
  memset(s, 0, sizeof(struct TcpSession));
  s->src = sourceAddress;
  s->sport = sourcePort;
  s->dst = targetAddress;
  s->dport = targetPort;
  s->snd_wnd = 0;
  s->snd_nxt = (uint32)mrand48();  /* random initial sequence number */
  s->iss = s->snd_nxt;
  s->rcv_nxt = 0;
  s->irs = 0;

  /* Now open a raw socket for sending our "fake" TCP segments */
  if ((rawSocket = socket(AF_INET, SOCK_RAW, IPPROTO_RAW)) < 0) {
    printf("ERROR: couldn't open socket\n");
    Quit();
  }
  if (setsockopt(rawSocket, IPPROTO_IP, IP_HDRINCL,
		 (char *)&flag,sizeof(flag)) < 0) {
    printf("ERROR: couldn't set raw socket options\n");
    Quit();
  }

  s->socket = rawSocket;

  /* 
   * Block kernel TCP from seeing any of the target's responses.
   * AFAIK its irrelvant that this is the same raw socket that we're
   * sending on.
   *
   * I wish that everyone would get their act together and support
   * ONE firewall API... imagine that.
   */

  memset(&firewallRule, 0, sizeof(firewallRule));

#ifdef __FreeBSD__
  firewallRule.fw_flg |= IP_FW_F_DENY | IP_FW_F_IN; 
  firewallRule.fw_prot = IPPROTO_TCP;
  firewallRule.fw_src.s_addr = s->dst;
  firewallRule.fw_smsk.s_addr = htonl(~0);
  firewallRule.fw_uar.fw_pts[0] = s->dport;
  IP_FW_SETNSRCP(&firewallRule, 1);
  firewallRule.fw_dst.s_addr = s->src;
  firewallRule.fw_uar.fw_pts[1] = s->sport;
  IP_FW_SETNDSTP(&firewallRule, 1);
  firewallRule.fw_dmsk.s_addr = htonl(~0);
  firewallRule.fw_number = 999;
  if (setsockopt(s->socket, IPPROTO_IP, IP_FW_ADD,
		 &firewallRule, sizeof(firewallRule)) != 0) {
    printf("ERROR: couldn't block kernel TCP for %s:%d\n",
	   InetAddress(s->dst), s->dport);
  }
#endif /* __FreeBSD__ */
#ifdef linux
  memcpy(firewallRule.fwc_label, IP_FW_LABEL_INPUT,
	 sizeof(firewallRule.fwc_label));
  memcpy(firewallRule.fwc_rule.label, IP_FW_LABEL_BLOCK,
	 sizeof(firewallRule.fwc_label)); 
  firewallRule.fwc_rule.ipfw.fw_proto = IPPROTO_TCP;

  firewallRule.fwc_rule.ipfw.fw_src.s_addr = s->dst;
  firewallRule.fwc_rule.ipfw.fw_smsk.s_addr = htonl(~0); 
  firewallRule.fwc_rule.ipfw.fw_spts[0] = s->dport;
  firewallRule.fwc_rule.ipfw.fw_spts[1] = s->dport;
  firewallRule.fwc_rule.ipfw.fw_dst.s_addr = s->src;
  firewallRule.fwc_rule.ipfw.fw_dmsk.s_addr = htonl(~0); 
  firewallRule.fwc_rule.ipfw.fw_dpts[0] = s->sport;
  firewallRule.fwc_rule.ipfw.fw_dpts[1] = s->sport;

  if (setsockopt(s->socket,IPPROTO_IP,IP_FW_APPEND,
		 &firewallRule,sizeof(firewallRule)) != 0) {
    printf("ERROR: couldn't block kernel TCP for %s:%d\n",
	   InetAddress(s->dst),  s->dport);
    Quit();
  }
#endif /* linux */
  initFirewall=1;

  SendSessionPacket(s, &synPacket, sizeof(struct TcpPacket),TCPFLAGS_SYN);   
  timeoutTime = GetTime() + SYNTIMEOUT;

  /* 
   * Wait for SYN/ACK and retransmit SYN if appropriate 
   * Kinda crappy, but it gets the job done 
   */
  while(!synAckReceived && numRetransmits < MAXSYNRETRANSMITS) {
    while(GetTime() < timeoutTime) {

      /* Have we captured any packets? */
      if ((p = (struct TcpPacket *)CaptureGetPacket(&pi)) == NULL) {
	continue;
      }

      /* Received a packet from us to them */
      if (INSESSION(p, s->src, s->sport, s->dst, s->dport)) {
	/* Is it a SYN? */
	if (p->tcp.tcp_flags == TCPFLAGS_SYN) {
	  ts1 = pi.ts.tv_sec + (double)pi.ts.tv_usec/1000000.0;
	  continue;
	}
	/* No? then its something weird */
	else {
	  printf("WARNING: unexpected packet during session establishment\n");
	  PrintTcpPacket(p);
	  continue;
	}
      }
      /* Next, check if a packet from them to us */
      else if (INSESSION(p, s->dst, s->dport, s->src, s->sport)) {
	/* Is it a SYN/ACK? */
	if (p->tcp.tcp_flags == (TCPFLAGS_SYN | TCPFLAGS_ACK)) {
	  timeoutTime = GetTime(); /* force exit */
	  synAckReceived++;
	  ts2 = pi.ts.tv_sec + (double)pi.ts.tv_usec/1000000.0;
	  if (numRetransmits > 0) {      
	    printf("(Unreliable) ");  /* ACK for which SYN? */
	  }
	  printf("Connection setup took %d ms\n",(int)((ts2-ts1)*1000.0));

	  /* Save ttl for,admittedly poor,indications of reverse route change */
	  s->ttl = p->ip.ip_ttl;
	  s->rcv_wnd = ntohl(p->tcp.tcp_win);
	}
	/* Is it a RST?  i.e., please stop */
	else if (p->tcp.tcp_flags & TCPFLAGS_RST) { 
	  printf("WARNING: Early Reset during session establishment\n");
	  Quit();
	}
	/* No? then its something weird */
	else {
	  printf("WARNING: unexpected packet during session establishment\n");
	  PrintTcpPacket(p);
	  continue;
	}
      }
      else {
	printf("WARNING: unexpected packet during session establishment\n");
	PrintTcpPacket(p);
	if (killOldFlows) {
	  /* If it was sent to us and wasn't a RST then tell them to stop */
	  if (p->ip.ip_dst == s->src && !(p->tcp.tcp_flags & TCPFLAGS_RST)) {
	    RespondWithReset(p);
	  }
	}
	continue;
      }
    }
    if (!synAckReceived) {
      if (verbose) {
	printf("SYN timeout.  Retransmitting\n");
      }
      SendSessionPacket(s, &synPacket, sizeof(struct TcpPacket),TCPFLAGS_SYN);
      timeoutTime = GetTime() + SYNTIMEOUT;
      numRetransmits++;
    }
  }
  if (numRetransmits >= MAXSYNRETRANSMITS) {
    printf("Could not establish contact after %d retries\n", numRetransmits);
    Quit();
  }
  
  /* Update session variables */
  s->irs = ntohl(p->tcp.tcp_seq);
  s->rcv_nxt = s->irs+1;  /* SYN/ACK takes up a byte of seq space */
  s->snd_nxt = s->iss+1;  /* SYN takes up a byte of seq space */
  s->snd_una = s->iss+1;  

  initSession = 1;

  printf("src = %s:%d (%u)\n", InetAddress(s->src),
	   s->sport, s->iss);
  printf("dst = %s:%d (%u)\n",InetAddress(s->dst),
	   s->dport, s->irs);

  return 1;
}


double *GenerateIntervals(int distribution, double mean, int count) 
{
  int i;
  double *intervals;

  if ((intervals=(double *)malloc(sizeof(double) * count)) == NULL) {
    printf("ERROR: couldn't malloc offset vector\n");
    return NULL;
  }
  
  for (i=0;i<count;i++) {
    switch(distribution) {
      case DISTRIBUTION_UNIFORM:
	intervals[i] = random_uniform(0, 2 * mean);
	break;
      case DISTRIBUTION_EXPONENTIAL:
	intervals[i] = random_exponential(mean);
	break;
      case DISTRIBUTION_PERIODIC:
        intervals[i] = mean;
	break;
      default:
        printf("ERROR: unknown distribution: %d\n", distribution);
	return NULL;
    }
  }
  return intervals;
}


double CheckIntervals(double *intervals, int distribution, double mean,
		      int count) 
{
  int i;
  double s;
  double *copy;

  /* the A2 tests mess with the data, so make a copy first */
  if ((copy = malloc(sizeof(double)*count)) == NULL) {
    printf("ERROR: out of memory\n");
    Quit();
  }
  for (i=0;i<count;i++) {
    copy[i] = intervals[i];
  }
#define EPSILON (0.01)  /* 10 ms */

  switch(distribution) {
     case DISTRIBUTION_UNIFORM:
        s = exp_A2_known_mean(copy, count, mean);
        break;
     case DISTRIBUTION_EXPONENTIAL:
        s = exp_A2_known_mean(copy, count, mean);
        break;
      case DISTRIBUTION_PERIODIC:
        s = unif_A2_known_range(copy, count, mean-EPSILON, mean+EPSILON);
        break;
      default:
        printf("ERROR: unknown distribution: %d\n", distribution);
	return -1.0;
    }
  return s;
}



int
GetCannonicalInfo(char *string, char name[MAXHOSTNAMELEN], uint32 *address)
{
  struct hostent *hp;

  /* Is string in dotted decimal format? */
  if ((*address = inet_addr(string)) == INADDR_NONE) {
    /* No, then lookup IP address */
    if ((hp = gethostbyname(string)) == NULL) {
      /* Can't find IP address */
      printf("ERROR: couldn't obtain address for %s\n", string);
      return -1;
    } else {
      strncpy(name, hp->h_name, MAXHOSTNAMELEN-1);
      memcpy((void *)address, (void *)hp->h_addr, hp->h_length);
    }
  } else {
    if ((hp = gethostbyaddr((char *)address, sizeof(*address),
			    AF_INET)) == NULL) {
      /* Can't get cannonical hostname, so just use input string */
      if (verbose) {
	printf("WARNING: couldn't obtain cannonical name for %s\n", string);
      }
      strncpy(name, string, MAXHOSTNAMELEN - 1);
    } else {
      strncpy(name, hp->h_name, MAXHOSTNAMELEN - 1);
    }
  }
  return 0;
}



int main(int argc, char **argv) 
{
  int count = DEFAULT_COUNT;
  int distribution = DEFAULT_DISTRIBUTION;
  double mean = DEFAULT_MEAN;
  uint16 targetPort = DEFAULT_TARGETPORT;
  uint16 sourcePort = 0;
  int packetSize = DEFAULT_PACKETSIZE;

  char targetHostName[MAXHOSTNAMELEN];	/* DNS name of target host */
  uint32 targetIpAddress;		/* IP address of target host */
  char sourceHostName[MAXHOSTNAMELEN];
  uint32 sourceIpAddress;
  char *target;
  char source[MAXHOSTNAMELEN];
  double *intervals;
  char *data=defaultRequest;
  char *dataBuffer;
  int dataSize;
  char *p;
  struct sockaddr_in saddr;
  struct stat statBuffer;
  int fd;
  int connectionWakeupTime = 0;
  int opt;
  RETSIGTYPE (*oldhandler)(int);
  
  
  if ((p = (char *)strrchr(argv[0], '/')) != NULL) {
    progName = p + 1;
  } else {
    progName = argv[0];
  }
  
  opterr = 0;
  while ((opt = getopt(argc, argv, "p:c:f:m:h:s:x:w:tv")) != EOF) {
    switch (opt) {
	case 'p':
           targetPort = atoi(optarg);
	   break;
	case 'w':
           sourcePort = atoi(optarg);
	   break;
	case 'c':
	   count = atoi(optarg);
	   break;
	case 'f':
           if (strcmp(optarg, "exponential") == 0) {
	     distribution = DISTRIBUTION_EXPONENTIAL;
	   } else if (strcmp(optarg, "uniform") == 0) {
	     distribution = DISTRIBUTION_UNIFORM;
	   } else if (strcmp(optarg, "periodic") == 0) {
	     distribution = DISTRIBUTION_PERIODIC;
	   } else {
	     Usage();
	   }
	   break;
	case 'm':
	   mean = atof(optarg);
	   break;
	case 's':     
           packetSize = atoi(optarg);
           if (packetSize < 41 || packetSize > 1460) {
              printf("Valid packet sizes are in the range 41-1460\n");
              Usage();
           }
	   /* XXX */
	   printf("WARNING: packet sizing does not work correctly in this release\n");
           break; 
	case 'v':
	   verbose = 1;
	   break;
	case 'h':     
	  if ((fd = open(optarg, O_RDONLY)) < 0) {
	    printf("ERROR: Could not open file %s\n",optarg);
	    Quit();
	  }
	  if (fstat(fd, &statBuffer) < 0) {
	    printf("ERROR: Could not stat file %s\n",optarg);
	    Quit();
	  }
	  if ((data = (char *)malloc(statBuffer.st_size + 1)) == NULL) {
	    printf("ERROR:Out of memory\n");
	    Quit();
	  }
	  if (read(fd, data, statBuffer.st_size) < 0) {
	    printf("ERROR: failure reading file %s\n",optarg);
	    Quit();
	  }
	  data[statBuffer.st_size] = '\0'; /* zero terminate for printing */
	  break;
        case 'x':
           connectionWakeupTime = atoi(optarg); 
           break;
	case 't':
	   test = 1;
	   break;
	default:
	   Usage();
    }
  }
  switch (argc - optind) {
      case 1:
         target = argv[optind];
	 break;
      default:
	 Usage();
  }

  /* Setup signal handler for cleaning up */
  (void)setsignal(SIGTERM, Cleanup);
  (void)setsignal(SIGINT, Cleanup);
  if ((oldhandler = setsignal(SIGHUP, Cleanup)) != SIG_DFL) 
    (void)setsignal(SIGHUP, oldhandler);

  /*
   * Get hostname and IP address of target host
   */
  if (GetCannonicalInfo(target, targetHostName, &targetIpAddress) < 0) {
    Quit();
  }

  /*
   * Get hostname and IP address of source host
   */
  if (gethostname(source, MAXHOSTNAMELEN) != 0) {
    printf("ERROR: can't determine local hostname\n");
    Quit();
  }
  if (GetCannonicalInfo(source, sourceHostName, &sourceIpAddress) < 0) {
    Quit();
  }

  if (verbose) {
    printf("source = %s [%s]\n", sourceHostName, InetAddress(sourceIpAddress));
    printf("target = %s [%s],\n",targetHostName, InetAddress(targetIpAddress));
    printf("port = %d, count = %d\n", targetPort, count);
    printf("distribution = %s, mean = %f seconds\n",
	   distributionNames[distribution], mean);
	   
  }

  /* Initialze random number generator */
  if (test) {
    srand48(test);
  } else {
    srand48(GetTime());
  }

  if (sourcePort == 0) {
    /* 
     * Find and allocate a spare TCP port to use
     */
    saddr.sin_family = AF_INET;
    if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
      printf("ERROR: can't open socket\n");
      return 0;
    }
    if ((sourcePort = BindTcpPort(fd)) == 0) {
      printf("ERROR: can't bind port\n");
      return 0;
    }
  }

  /* Init packet capture device and install filter for our flow */
  CaptureInit(sourceIpAddress, sourcePort, targetIpAddress, targetPort);
  initCapture = 1;

  /* Setup connection to target */
  if ( EstablishSession(sourceIpAddress, sourcePort,
			 targetIpAddress, targetPort, &session) == 0) {
    printf("ERROR: Couldn't establish session to %s:%d\n",
	   targetHostName,targetPort);
    Quit();
  }

  if ((intervals = GenerateIntervals(distribution, mean, count)) == NULL) {
    printf("ERROR: couldn't generate sampling distribution\n");
    Quit();
  }

  /* one packet's data portion + one pyte per packet + artificial hole */
  dataSize = packetSize-sizeof(struct TcpPacket)+count*sizeof(uint8)+SEQSKIP;
  if ((dataBuffer = (char *)malloc(dataSize)) == NULL) {
    printf("ERROR: out of memory\n");
    Quit();
  }
  memset(dataBuffer, 0, dataSize);
  memcpy((void *)dataBuffer,(void *)data,
	 MIN(strlen(data),dataSize)); 

  DataSeeding(&session, dataBuffer, packetSize, count, intervals);   
  HoleFilling(&session, dataBuffer, packetSize);  

  if (verbose) {
    printf("Wire time consistent with distribution (significance = %f)\n",
	   CheckIntervals(intervals, distribution, mean, count-1));
  }    

  dataReceived = dataSent - dataLost;
  acksSent = dataReceived;
  if (acksSent < acksReceived) {
    printf("WARNING: possible remote buffer overrun... \n");
  }

  printf("dataSent = %d, dataReceived = %d", dataSent, dataReceived);
  printf("\n");
  printf("acksSent = %d, acksReceived = %d", acksSent, acksReceived);
  printf("\n");

  if (dataSent > 0) {
    printf("Forward drop rate = %f\n",1-(float)dataReceived/(float)dataSent);
  }
  if (acksSent > 0) {
    printf("Reverse drop rate = %f\n",1-(float)acksReceived/(float)acksSent);
  }
  Cleanup(1);
  close(session.socket); 
  return(0);
}




