/*
 * Copyright (c) 1998, 1999
 * Stefan Savage and the University of Washington.
 * All rights reserved. 
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author(s) may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.  
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR(S) ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR(S) BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * HISTORY
 * 04-Nov-99  Stefan Savage (savage) at the University of Washington
 *	Linux vs FreeBSD interpretation of SOCK_RAW IP header byte-order
 *
 * 21-Mar-99  Stefan Savage (savage) at the University of Washington
 *	Added session support
 *
 * 11-Aug-98  Stefan Savage (savage) at the University of Washington
 *	Added ReadTcpPacket and PrintTcpPacket
 *
 * 06-Aug-98  Stefan Savage (savage) at the University of Washington
 *	Created
 *
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h> 
#include <string.h>
#include <stdio.h>
#include "inet.h"
/*
 * Deal with struct in_addr type agreement once and for all
 */
char *InetAddress(uint32 addr)
{
  struct in_addr s;
  s.s_addr = addr;
  return (inet_ntoa(s));
}

/*
 * Really slow implementation of ip checksum
 * ripped off from rfc1071
 */
uint16 InetChecksum(uint16 *addr, uint16 len){
  uint32 sum = 0;
  uint32 count = len;

  while( count > 1 )  {
    sum += *addr++;
    count -= 2;
  }
  if( count > 0 ) {
    sum += * (uint8 *) addr;
  }

  while (sum>>16) {
    sum = (sum & 0xffff) + (sum >> 16);
  }
  return(~sum);
}


void WriteTcpPacket(struct TcpPacket *p,
		    uint32 src, uint32 dst, uint16 sport, uint16 dport,
		    uint32 seq, uint32 ack, uint8 flags, uint16 win,
		    uint16 urp, uint16 datalen, uint16 optlen) 
{
  struct TcpHeader *tcp = &p->tcp;
  struct IpHeader *ip = &p->ip;

  bzero((char *)p, sizeof (struct IpHeader)); 
  ip->ip_src = src;
  ip->ip_dst = dst;
  ip->ip_p = IPPROTOCOL_TCP;
  ip->ip_xsum =
    htons((uint16)(sizeof(struct TcpHeader)+datalen+optlen)); /* pseudo hdr */
  tcp->tcp_sport = htons(sport);
  tcp->tcp_dport = htons(dport);
  tcp->tcp_seq = htonl(seq);
  tcp->tcp_ack = htonl(ack);
  tcp->tcp_hl = (sizeof (struct TcpHeader) + optlen) << 2;
  tcp->tcp_flags = flags;
  tcp->tcp_win = htons(win);
  tcp->tcp_urp = htons(urp);
  tcp->tcp_xsum = 0;
  tcp->tcp_xsum = InetChecksum((uint16 *)ip,
			       (uint16)(sizeof(struct TcpHeader) +
					sizeof(struct IpHeader)+
					datalen+optlen));


  /* Fill in real ip header */
  ip->ip_ttl = 60;
  ip->ip_tos = 0;
  ip->ip_vhl = 0x45;
  ip->ip_p = IPPROTOCOL_TCP;
#ifdef __FreeBSD__
  ip->ip_off = IP_DF;
  ip->ip_len = (uint16)(sizeof (struct TcpPacket) + datalen + optlen);
#else /* __FreeBSD__ */
  ip->ip_off = htons(IP_DF);
  ip->ip_len = htons((uint16)((sizeof (struct TcpPacket) + datalen + optlen)));
#endif /* __FreeBSD__ */

}


void ReadTcpPacket(struct TcpPacket *p,
		    uint32 *src, uint32 *dst, uint16 *sport, uint16 *dport,
		    uint32 *seq, uint32 *ack, uint8 *flags, uint16 *win,
		    uint16 *urp, uint16 *datalen, uint16 *optlen) 
{
  struct TcpHeader *tcp = &p->tcp;
  struct IpHeader *ip = &p->ip;

  uint16 ip_len;
  uint16 ip_hl;
  uint16 tcp_hl;

  /* XXX do checksum check? */
  if (ip->ip_p != IPPROTOCOL_TCP) {
    printf("Error: not a TCP packet\n");
    return;
  }

  *src = ip->ip_src;
  *dst = ip->ip_dst;
  *sport = ntohs(tcp->tcp_sport);
  *dport = ntohs(tcp->tcp_dport);
  *seq = ntohl(tcp->tcp_seq);
  *ack = ntohl(tcp->tcp_ack);
  *flags = tcp->tcp_flags;
  *win = ntohs(tcp->tcp_win);
  *urp = ntohs(tcp->tcp_urp);

  tcp_hl = tcp->tcp_hl >> 2;
  ip_len = ip->ip_len;
  ip_hl = (ip->ip_vhl & 0x0f) << 2;
  *datalen = (ip_len - ip_hl) - tcp_hl;
  *optlen = tcp_hl - sizeof(struct TcpHeader);
}

 
void PrintTcpPacket(struct TcpPacket *p)
{
  struct TcpHeader *tcp = &p->tcp;
  struct IpHeader *ip = &p->ip;

  printf("%s.%u > ",InetAddress(ip->ip_src), ntohs(tcp->tcp_sport));
  printf("%s.%u ", InetAddress(ip->ip_dst), ntohs(tcp->tcp_dport));
  if (tcp->tcp_flags & TCPFLAGS_FIN) {
    printf("F");
  }
  if (tcp->tcp_flags & TCPFLAGS_SYN) {
    printf("S");
  }
  if (tcp->tcp_flags & TCPFLAGS_RST) {
    printf("R");
  }
  if (tcp->tcp_flags & TCPFLAGS_PSH) {
    printf("P");
  }
  if (tcp->tcp_flags & TCPFLAGS_ACK) {
    printf("A");
  }
  if (tcp->tcp_flags & TCPFLAGS_URG) {
    printf("U");
  }
  printf(" seq: %u, ack: %u",
	 (uint32)ntohl(tcp->tcp_seq),
	 (uint32)ntohl(tcp->tcp_ack));
  printf(" win: %u, urg: %u",ntohs(tcp->tcp_win), ntohs(tcp->tcp_urp));
  printf(" datalen: %u, optlen: %u\n",
	 ntohs(ip->ip_len) - ((ip->ip_vhl &0x0f) << 2) - (tcp->tcp_hl >> 2),
	 (tcp->tcp_hl >> 2) - (unsigned int)sizeof (struct TcpHeader));

}


void SendSessionPacket(struct TcpSession *s, struct TcpPacket *p,
		       uint16 ip_len, uint8 tcp_flags)
{
  int nbytes;
  struct sockaddr_in sockAddr;   

  (void) WriteTcpPacket(p,
			s->src,
			s->dst,
			s->sport,
			s->dport,
			s->snd_nxt, 
			s->rcv_nxt, 
			tcp_flags,
			s->snd_wnd,
			0, ip_len - sizeof(struct TcpPacket), 0);

  sockAddr.sin_family  = AF_INET;
  sockAddr.sin_addr.s_addr = s->dst;

  if ((nbytes = sendto(s->socket, (char *)p, ip_len, 0,
		       (struct sockaddr *)&sockAddr,
		       sizeof(sockAddr))) < ip_len) {
    printf("WARNING: only sent %d of %d bytes\n", nbytes, ip_len);
  }
}

