/* packet.h
 * Definitions for packet disassembly structures and routines
 *
 * Ethereal - Network traffic analyzer
 * By Gerald Combs <gerald@zing.org>
 * Copyright 1998 Gerald Combs
 *
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */


#ifndef __PACKET_H__
#define __PACKET_H__


#define pntohs(x)	*((guint16*) (x) )
#define pntohl(x)	*((guint32*) (x) )


#define IEEE_802_3_MAX_LEN 1500
#define BYTE_VIEW_WIDTH    16

typedef struct _frame_data {
  guint32  pkt_len;         /* Packet length */
  guint32  cap_len;         /* Amount actually captured */
  guint32  secs;            /* Seconds */
  guint32  usecs;           /* Microseconds */
  long     file_off;        /* File offset */
  gchar   *win_info[5];        /* Packet list text */
} frame_data;

/* Many of the structs and definitions below were taken from include files
 * in the Linux distribution. */

/* ARP / RARP structs and definitions */

typedef struct _e_ether_arp {
  guint16 ar_hrd;
  guint16 ar_pro;
  guint8  ar_hln;
  guint8  ar_pln;
  guint16 ar_op;
  guint8  arp_sha[6];
  guint8  arp_spa[4];
  guint8  arp_tha[6];
  guint8  arp_tpa[4];
} e_ether_arp;

#ifndef ARPOP_REQUEST
#define ARPOP_REQUEST  1       /* ARP request.  */
#endif
#ifndef ARPOP_REPLY
#define ARPOP_REPLY    2       /* ARP reply.  */
#endif
/* Some OSes have different names, or don't define these at all */
#ifndef ARPOP_RREQUEST
#define ARPOP_RREQUEST 3       /* RARP request.  */
#endif
#ifndef ARPOP_RREPLY
#define ARPOP_RREPLY   4       /* RARP reply.  */
#endif

/* ICMP structs and definitions */

typedef struct _e_icmp {
  guint8  icmp_type;
  guint8  icmp_code;
  guint16 icmp_cksum;
  union {
    struct {  /* Address mask request/reply */
      guint16 id;
      guint16 seq;
      guint32 sn_mask;
    } am;
    struct {  /* Timestap request/reply */
      guint16 id;
      guint16 seq;
      guint32 orig;
      guint32 recv;
      guint32 xmit;
    } ts;
    guint32 zero;  /* Unreachable */
  } opt;
} e_icmp;

#define ICMP_ECHOREPLY     0
#define ICMP_UNREACH       3
#define ICMP_SOURCEQUENCH  4
#define ICMP_REDIRECT      5
#define ICMP_ECHO          8
#define ICMP_TIMXCEED     11
#define ICMP_PARAMPROB    12
#define ICMP_TSTAMP       13
#define ICMP_TSTAMPREPLY  14
#define ICMP_IREQ         15
#define ICMP_IREQREPLY    16
#define ICMP_MASKREQ      17
#define ICMP_MASKREPLY    18

/* IGMP structs and definitions */

typedef struct _e_igmp {
#if BYTE_ORDER == BIG_ENDIAN
  guint8  igmp_v:4;
  guint8  igmp_t:4;
#else /* Little endian */
  guint8  igmp_t:4;
  guint8  igmp_v:4;
#endif
  guint8  igmp_unused;
  guint16 igmp_cksum;
  guint32 igmp_gaddr;
} e_igmp;

#define IGMP_M_QRY     0x01
#define IGMP_V1_M_RPT  0x02
#define IGMP_V2_LV_GRP 0x07
#define IGMP_DVMRP     0x03
#define IGMP_PIM       0x04
#define IGMP_V2_M_RPT  0x06
#define IGMP_MTRC_RESP 0x1e
#define IGMP_MTRC      0x1f

/* IP structs and definitions */

typedef struct _e_ip {
#if BYTE_ORDER == BIG_ENDIAN
  guint8  ip_v:4;
  guint8  ip_hl:4;
#else /* Little endian */
  guint8  ip_hl:4;
  guint8  ip_v:4;
#endif
  guint8  ip_tos;
  guint16 ip_len;
  guint16 ip_id;
  guint16 ip_off;
  guint8  ip_ttl;
  guint8  ip_p;
  guint16 ip_sum;
  guint32 ip_src;
  guint32 ip_dst;
} e_ip;

#define IPTOS_TOS_MASK    0x1E
#define IPTOS_TOS(tos)    ((tos) & IPTOS_TOS_MASK)
#define IPTOS_NONE        0x00
#define IPTOS_LOWDELAY    0x10
#define IPTOS_THROUGHPUT  0x08
#define IPTOS_RELIABILITY 0x04
#define IPTOS_LOWCOST     0x02

#define IP_PROTO_ICMP  1
#define IP_PROTO_IGMP  2
#define IP_PROTO_TCP   6
#define IP_PROTO_UDP  17
#define IP_PROTO_OSPF 89

/* TCP structs and definitions */

typedef struct _e_tcphdr {
  guint16 th_sport;
  guint16 th_dport;
  guint32 th_seq;
  guint32 th_ack;
#if BYTE_ORDER == LITTLE_ENDIAN
  guint8  th_x2:4;
  guint8  th_off:4;
#else
  guint8  th_off:4;
  guint8  th_x2:4;
#endif
  guint8  th_flags;
#define TH_FIN  0x01
#define TH_SYN  0x02
#define TH_RST  0x04
#define TH_PUSH 0x08
#define TH_ACK  0x10
#define TH_URG  0x20
  guint16 th_win;
  guint16 th_sum;
  guint16 th_urp;
} e_tcphdr;

/* UDP structs and definitions */

typedef struct _e_udphdr {
  guint16 uh_sport;
  guint16 uh_dport;
  guint16 uh_ulen;
  guint16 uh_sum;
} e_udphdr;

/* Ports */

#define PORT_DNS 53

/* Utility routines used by packet*.c */
gchar*     ether_to_str(guint8 *);
gchar*     ip_to_str(guint8 *);
void       packet_hex_print(GtkText *, guint8 *, gint, gint, gint);
GtkWidget* add_item_to_tree(GtkWidget *, gint, gint, gchar *, ...);
void       decode_start_len(GtkTreeItem *, gint*, gint*);

/* Routines in packet.c */
void dissect_packet(const u_char *, frame_data *, GtkTree *);

/*
 * Routines in packet-*.c
 * Routines should take three args: packet data *, frame_data *, tree *
 * They should never modify the packet data.
 */
void dissect_eth(const u_char *, frame_data *, GtkTree *);
void dissect_ppp(const u_char *, frame_data *, GtkTree *);
void dissect_tr(const u_char *, frame_data *, GtkTree *);

/*
 * Routines in packet-*.c
 * Routines should take four args: packet data *, offset, frame_data *,
 * tree *
 * They should never modify the packet data.
 */
void dissect_arp(const u_char *, int, frame_data *, GtkTree *);
void dissect_dns(const u_char *, int, frame_data *, GtkTree *);
void dissect_icmp(const u_char *, int, frame_data *, GtkTree *);
void dissect_igmp(const u_char *, int, frame_data *, GtkTree *);
void dissect_ip(const u_char *, int, frame_data *, GtkTree *);
void dissect_ipx(const u_char *, int, frame_data *, GtkTree *);
void dissect_llc(const u_char *, int, frame_data *, GtkTree *);
void dissect_ospf(const u_char *, int, frame_data *, GtkTree *);
void dissect_ospf_hello(const u_char *, int, frame_data *, GtkTree *);
void dissect_raw(const u_char *, int, frame_data *, GtkTree *);
void dissect_tcp(const u_char *, int, frame_data *, GtkTree *);
void dissect_trmac(const u_char *, int, frame_data *, GtkTree *);
void dissect_udp(const u_char *, int, frame_data *, GtkTree *);

/* This function is in ethertype.c */
void ethertype(guint16 etype, int offset,
		const u_char *pd, frame_data *fd, GtkTree *tree,
		GtkWidget *fh_tree);

#endif /* packet.h */
