/* packet.c
 * Routines for packet disassembly
 *
 * Ethereal - Network traffic analyzer
 * By Gerald Combs <gerald@unicom.net>
 * Copyright 1998 Gerald Combs
 *
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>
#include <netinet/in.h>

/*
#ifdef HAVE_NET_ETHERNET_H
  #include <net/ethernet.h>
#else
  #include <sys/ethernet.h>
#endif
*/

#include "packet.h"
#include "ethereal.h"
#include "etypes.h"

extern GtkWidget *byte_view;
extern GdkFont   *m_r_font, *m_b_font;

gchar *
ether_to_str(guint8 *ad) {
  static gchar  str[3][18];
  static gchar *cur;

  if (cur == &str[0][0]) {
    cur = &str[1][0];
  } else if (cur == &str[1][0]) {  
    cur = &str[2][0];
  } else {  
    cur = &str[0][0];
  }
  sprintf(cur, "%02x:%02x:%02x:%02x:%02x:%02x", ad[0], ad[1], ad[2],
    ad[3], ad[4], ad[5]);
  return cur;
}

gchar *
ip_to_str(guint8 *ad) {
  static gchar  str[3][16];
  static gchar *cur;

  if (cur == &str[0][0]) {
    cur = &str[1][0];
  } else if (cur == &str[1][0]) {  
    cur = &str[2][0];
  } else {  
    cur = &str[0][0];
  }
  sprintf(cur, "%d.%d.%d.%d", ad[0], ad[1], ad[2], ad[3]);
  return cur;
}

void
packet_hex_print(GtkText *bv, guchar *pd, gint len, gint bstart, gint blen) {
  gint     i = 0, j, k, cur;
  gchar    line[128], hexchars[] = "0123456789abcdef";
  GdkFont *cur_font, *new_font;
  
  while (i < len) {
    /* Print the line number */
    sprintf(line, "%04x  ", i);
    gtk_text_insert(bv, m_r_font, NULL, NULL, line, -1);
    /* Do we start in bold? */
    cur_font = (i >= bstart && i < (bstart + blen)) ? m_b_font : m_r_font;
    j   = i;
    k   = i + BYTE_VIEW_WIDTH;
    cur = 0;
    /* Print the hex bit */
    while (i < k) {
      if (i < len) {
        line[cur++] = hexchars[(pd[i] & 0xf0) >> 4];
        line[cur++] = hexchars[pd[i] & 0x0f];
      } else {
        line[cur++] = ' '; line[cur++] = ' ';
      }
      line[cur++] = ' ';
      i++;
      /* Did we cross a bold/plain boundary? */
      new_font = (i >= bstart && i < (bstart + blen)) ? m_b_font : m_r_font;
      if (cur_font != new_font) {
        gtk_text_insert(bv, cur_font, NULL, NULL, line, cur);
        cur_font = new_font;
        cur = 0;
      }
    }
    line[cur++] = ' ';
    gtk_text_insert(bv, cur_font, NULL, NULL, line, cur);
    cur = 0;
    i = j;
    /* Print the ASCII bit */
    cur_font = (i >= bstart && i < (bstart + blen)) ? m_b_font : m_r_font;
    while (i < k) {
      if (i < len) {
        line[cur++] = (isgraph(pd[i])) ? pd[i] : '.';
      } else {
        line[cur++] = ' ';
      }
      i++;
      /* Did we cross a bold/plain boundary? */
      new_font = (i >= bstart && i < (bstart + blen)) ? m_b_font : m_r_font;
      if (cur_font != new_font) {
        gtk_text_insert(bv, cur_font, NULL, NULL, line, cur);
        cur_font = new_font;
        cur = 0;
      }
    }
    line[cur++] = '\n';
    line[cur]   = '\0';
    gtk_text_insert(bv, cur_font, NULL, NULL, line, -1);
  }
}

GtkWidget *
add_item_to_tree(GtkWidget *tree, gint start, gint len,
  gchar *format, ...) {
  GtkWidget *ti;
  va_list    ap;
  gchar      label_str[256];
  guint32    t_info;
  
  /* This limits us to a max packet size of 65535 bytes. */
  t_info = ((start & 0xffff) << 16) | (len & 0xffff);
  va_start(ap, format);
  vsnprintf(label_str, 256, format, ap);
  ti = gtk_tree_item_new_with_label(label_str);
  gtk_object_set_user_data(GTK_OBJECT(ti), (gpointer) t_info);
  gtk_tree_append(GTK_TREE(tree), ti);
  gtk_widget_show(ti);

  return ti;
}

void
dissect_packet(const u_char *pd, frame_data *fd, GtkTree *tree) {
  guint16    etype, length;
  gchar      etype_str[][10] = {"IP", "ARP", "RARP", "AppleTalk", "AARP"};
  int        offset = 14;
  GtkWidget *fh_tree, *ti;

  if (fd->win_info[0]) {
    strcpy(fd->win_info[2], ether_to_str((guint8 *)&pd[0]));
    strcpy(fd->win_info[1], ether_to_str((guint8 *)&pd[6]));
    strcpy(fd->win_info[4], "Ethernet II");
  }

  etype = (pd[12] << 8) | pd[13];
  if (etype <= IEEE_802_3_MAX_LEN) {
    length = etype;
    /* To do: check for 802.2 LLC and SNAP info */
    etype  = (pd[20] << 8) | pd[21];
    if (fd->win_info[0]) { sprintf(fd->win_info[4], "802.3 (0x%04x)", etype); }
    if (tree) {
      ti = add_item_to_tree(GTK_WIDGET(tree), 0, 22,
        "IEEE 802.3 (%d on wire, %d captured)", fd->pkt_len, fd->cap_len);
      fh_tree = gtk_tree_new();
      gtk_tree_item_set_subtree(GTK_TREE_ITEM(ti), fh_tree);
      add_item_to_tree(fh_tree, 0, 6, "Destination: %s",
        ether_to_str((guint8 *) &pd[0]));
      add_item_to_tree(fh_tree, 6, 6, "Source: %s",
        ether_to_str((guint8 *) &pd[6]));
      add_item_to_tree(fh_tree, 12, 2, "Length: %d", length);
      add_item_to_tree(fh_tree, 14, 3, "802.2 LLC: 0x%02x/0x%02x/0x%02x",
        pd[14], pd[15], pd[16]);
    }
    offset = 22;
  } else if (tree) {
    ti = add_item_to_tree(GTK_WIDGET(tree), 0, 14,
      "Ethernet II (%d on wire, %d captured)", fd->pkt_len, fd->cap_len);
    fh_tree = gtk_tree_new();
    gtk_tree_item_set_subtree(GTK_TREE_ITEM(ti), fh_tree);
    add_item_to_tree(fh_tree, 0, 6, "Destination: %s",
      ether_to_str((guint8 *) &pd[0]));
    add_item_to_tree(fh_tree, 6, 6, "Source: %s",
      ether_to_str((guint8 *) &pd[6]));
  }
  
  switch (etype) {
    case ETHERTYPE_IP:
      if (tree) {
        add_item_to_tree(fh_tree, offset - 2, 2, "Type: IP (0x%04x)",
          etype);
      }
      dissect_ip(pd, offset, fd, tree);
      break;
    case ETHERTYPE_ARP:
      if (tree) {
        add_item_to_tree(fh_tree, offset - 2, 2,
          "Type: ARP (0x%04x)", etype);
      }
      dissect_arp(pd, offset, fd, tree);
      break;
    case ETHERTYPE_REVARP:
      if (tree) {
        add_item_to_tree(fh_tree, offset - 2, 2,
          "Type: RARP (0x%04x)", etype);
      }
      dissect_arp(pd, offset, fd, tree);
      break;
    case ETHERTYPE_ATALK:
      if (tree) {
        add_item_to_tree(fh_tree, offset - 2, 2,
          "Type: AppleTalk (0x%04x)", etype);
      }
      if (fd->win_info[0]) { strcpy(fd->win_info[3], etype_str[3]); }
      break;
    case ETHERTYPE_AARP:
      if (tree) {
        add_item_to_tree(fh_tree, offset - 2, 2,
          "Type: AARP (0x%04x)", etype);
      }
      if (fd->win_info[0]) { strcpy(fd->win_info[3], etype_str[4]); }
      break;
    default:
      if (tree)
        add_item_to_tree(fh_tree, offset - 2, 2,
          "Type: Unknown (0x%04x)", etype);
      if (fd->win_info[0]) { sprintf(fd->win_info[3], "0x%04x", etype); }
      break;
  }
}
