/* packet-ip.c
 * Routines for IP layer packet disassembly
 *
 * Ethereal - Network traffic analyzer
 * By Gerald Combs <gerald@unicom.net>
 * Copyright 1998 Gerald Combs
 *
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "config.h"
#include <pcap.h>

#include <gtk/gtk.h>

#include <stdio.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/tcp.h>

#include "packet.h"
#include "ethereal.h"
#include "etypes.h"

void
dissect_ip(const u_char *pd, int offset, frame_data *fd, GtkTree *tree) {
  struct ip  iph;
  GtkWidget *ip_tree, *ti;

  /* To do: check for runts, errs, etc. */
  /* Avoids alignment problems on many architectures. */
  memcpy(&iph, &pd[offset], sizeof(struct ip));
  iph.ip_len = ntohs(iph.ip_len);
  iph.ip_id  = ntohs(iph.ip_id);
  iph.ip_off = ntohs(iph.ip_off);
  iph.ip_sum = ntohs(iph.ip_sum);
  
  if (fd->win_info[0]) {
    switch (iph.ip_p) {
      case 1:
        strcpy(fd->win_info[3], "ICMP");
        break;
      case 2:
        strcpy(fd->win_info[3], "IGMP");
        break;
      default:
        strcpy(fd->win_info[3], "IP");
        sprintf(fd->win_info[4], "Unknown IP protocol (%02x)", iph.ip_p);
    }

    strcpy(fd->win_info[1], ip_to_str((guint8 *) &iph.ip_src));
    strcpy(fd->win_info[2], ip_to_str((guint8 *) &iph.ip_dst));
  }
  
  if (tree) {
    ti = add_item_to_tree(GTK_WIDGET(tree), offset, (iph.ip_hl * 4),
      "Internet Protocol");
    ip_tree = gtk_tree_new();
    gtk_tree_item_set_subtree(GTK_TREE_ITEM(ti), ip_tree);
    add_item_to_tree(ip_tree, offset,      1, "Version: %d", iph.ip_v);
    add_item_to_tree(ip_tree, offset,      1, "Header length: %d", iph.ip_hl); 
    add_item_to_tree(ip_tree, offset +  1, 1, "Type of service: 0x%02x",
      iph.ip_tos);
    add_item_to_tree(ip_tree, offset +  2, 2, "Total length: %d", iph.ip_len);
    add_item_to_tree(ip_tree, offset +  4, 2, "Identification: 0x%04x",
      iph.ip_id);
    /* To do: add flags */
    add_item_to_tree(ip_tree, offset +  6, 2, "Fragment offset: %d",
      iph.ip_off & 0x1fff);
    add_item_to_tree(ip_tree, offset +  8, 1, "Time to live: %d",
      iph.ip_ttl);
    add_item_to_tree(ip_tree, offset +  9, 1, "Protocol: 0x%02x",
      iph.ip_p);
    add_item_to_tree(ip_tree, offset + 10, 2, "Header checksum: 0x%04x",
      iph.ip_sum);
    add_item_to_tree(ip_tree, offset + 12, 4, "Source address: %s",
      ip_to_str((guint8 *) &iph.ip_src));
    add_item_to_tree(ip_tree, offset + 16, 4, "Destination address: %s",
      ip_to_str((guint8 *) &iph.ip_dst));
  }

  switch (iph.ip_p) {
    case 6:
      dissect_tcp(pd, offset + (iph.ip_hl * 4), fd, tree);
     break;
   case 17:
      dissect_udp(pd, offset + (iph.ip_hl * 4), fd, tree);
      break;
  }
}
