/* packet-arp.c
 * Routines for ARP packet disassembly
 *
 * Ethereal - Network traffic analyzer
 * By Gerald Combs <gerald@unicom.net>
 * Copyright 1998 Gerald Combs
 *
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <gtk/gtk.h>

#include <stdio.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>

#include <pcap.h>

#include "packet.h"
#include "ethereal.h"
#include "etypes.h"

/* Some OSes have different names, or don't define these at all */
#ifndef ARPOP_RREQUEST
#define ARPOP_RREQUEST  3       /* RARP request.  */
#endif
#ifndef ARPOP_RREPLY
#define ARPOP_RREPLY    4       /* RARP reply.  */
#endif

void
dissect_arp(const u_char *pd, int offset, frame_data *fd, GtkTree *tree) {
  struct ether_arp *ea;
  guint16           hard_fmt, prot_fmt, opcode;
  gchar             *req_type[] = { "ARP request", "ARP reply",
                    "RARP request", "RARP reply" };
  GtkWidget        *arp_tree, *ti;

  /* To do: Check for {cap len,pkt len} < struct len */
  ea = (struct ether_arp *) &pd[offset];
  hard_fmt = ntohs(ea->ea_hdr.ar_hrd);
  prot_fmt = ntohs(ea->ea_hdr.ar_pro);
  opcode   = ntohs(ea->ea_hdr.ar_op);
  
  if (fd->win_info[0]) { strcpy(fd->win_info[3], "ARP"); }
  
  if (tree) {
    ti = add_item_to_tree(GTK_WIDGET(tree), offset, 28, req_type[opcode - 1]);
    arp_tree = gtk_tree_new();
    gtk_tree_item_set_subtree(GTK_TREE_ITEM(ti), arp_tree);
    add_item_to_tree(arp_tree, offset,      2,
      "Hardware type: 0x%04x", hard_fmt);
    add_item_to_tree(arp_tree, offset +  2, 2,
      "Protocol type: 0x%04x", hard_fmt);
    add_item_to_tree(arp_tree, offset +  4, 1,
      "Hardware size: 0x%02x", hard_fmt);
    add_item_to_tree(arp_tree, offset +  5, 1,
      "Protocol size: 0x%02x", hard_fmt);
    add_item_to_tree(arp_tree, offset +  6, 2,
      "Opcode: 0x%04x", hard_fmt);
    add_item_to_tree(arp_tree, offset +  8, 6,
      "Sender ether: %s", ether_to_str((guint8 *) ea->arp_sha));
    add_item_to_tree(arp_tree, offset + 14, 4,
      "Sender IP: %s", ip_to_str((guint8 *) ea->arp_spa));
    add_item_to_tree(arp_tree, offset + 18, 6,
      "Target ether: %s", ether_to_str((guint8 *) ea->arp_tha));
    add_item_to_tree(arp_tree, offset + 24, 4,
      "Target IP: %s", ip_to_str((guint8 *) ea->arp_tpa));
  }

  if (prot_fmt != ETHERTYPE_IP && fd->win_info[0]) {
    sprintf(fd->win_info[4], "h/w %d (%d) prot %d (%d) op 0x%04x",
      hard_fmt, ea->ea_hdr.ar_hln, prot_fmt, ea->ea_hdr.ar_pln, opcode);
    return;
  }
  switch (opcode) {
    case ARPOP_REQUEST:
      if (fd->win_info[0]) {
        sprintf(fd->win_info[4], "Who has %s?  Tell %s",
          ip_to_str((guint8 *) ea->arp_tpa), ip_to_str((guint8 *) ea->arp_spa));
      }
      break;
    case ARPOP_REPLY:
      if (fd->win_info[0]) {
        sprintf(fd->win_info[4], "%s is at %s",
          ip_to_str((guint8 *) ea->arp_spa),
          ether_to_str((guint8 *) ea->arp_sha));
      }
      break;
    case ARPOP_RREQUEST:
      if (fd->win_info[0]) {
        strcpy(fd->win_info[3], "RARP");
        sprintf(fd->win_info[4], "Who is %s?  Tell %s",
          ether_to_str((guint8 *) ea->arp_tha), 
          ether_to_str((guint8 *) ea->arp_sha));
      }
      break;
    case ARPOP_RREPLY:
      if (fd->win_info[0]) {
        strcpy(fd->win_info[3], "RARP");
        sprintf(fd->win_info[4], "%s is at %s",
          ether_to_str((guint8 *) ea->arp_sha),
          ip_to_str((guint8 *) ea->arp_spa));
      }
      break;
  }
}
