/* packet-postgresql.c
 * Routines for postgresql packet disassembly
 *
 * Copyright 2004, Edwin Calo <calo@fusemail.com>
 *
 * $Id: packet-postgresql.c 11410 2004-07-18 18:06:47Z gram $
 *
 * Ethereal - Network traffic analyzer
 * By Gerald Combs <gerald@ethereal.com>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>
#include <epan/packet.h>

static int proto_postgresql = -1;
static int hf_postgresql_response = -1;
static int hf_postgresql_request = -1;
static int hf_postgresql_length = -1;
static int hf_postgresql_string_size = -1;
static int hf_postgresql_string = -1;
static int hf_postgresql_total_length = -1;
static int hf_postgresql_bitone = -1;
static int hf_postgresql_buff_remaining = -1;
static int hf_postgresql_opcode = -1;
static int hf_postgresql_idone = -1;
static gint ett_postgresql = -1;

#define TCP_PORT_POSTGRESQL	5432

/*
 * For a protocol description, see:
 *
 *	http://www.postgresql.org/docs/7.4/interactive/protocol.html
 *
 * for version 3.0 of the protocol and
 *
 *	http://www.postgresql.org/docs/7.3/interactive/protocol.html
 *
 * for version 2.0 of the protocol and
 *
 *	http://www.postgresql.org/docs/6.3/interactive/c50.htm
 *
 * for version 1.0 of the protocol.
 *
 * It looks like there are two types of PDUs: Old style and new style.
 * New style PDUs start with a Length guint32, where the high byte is
 * equal to 0. Old style PDUs start with a single Byte (different from 0)
 * that contains the type of the PDU.
 *
 * The specific types can be found at:
 * http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html
 * 
 */

static void
dissect_postgresql (tvbuff_t * tvb, packet_info * pinfo, proto_tree * tree)
{
  proto_tree *postgresql_tree;
  proto_item *ti;
  gint offset = 0;
  gint buff_remaining = 0;

  guint8 *string;
  guint8 bitone;
  gint flag = 0;
  gint counter = 0;




  if (check_col (pinfo->cinfo, COL_PROTOCOL))
    col_set_str (pinfo->cinfo, COL_PROTOCOL, "POSTGRESQL");

  ti = proto_tree_add_item (tree, proto_postgresql, tvb, offset, -1, FALSE);
  postgresql_tree = proto_item_add_subtree (ti, ett_postgresql);

  buff_remaining = tvb_length_remaining (tvb, offset);

  if (check_col (pinfo->cinfo, COL_INFO))
    {
      col_add_str (pinfo->cinfo, COL_INFO,
		   (pinfo->match_port ==
		    pinfo->destport) ? " Request" : " Response");
    }

    counter=0;
    flag=0;
    while ( buff_remaining >= 1 )
    {
         bitone = tvb_get_guint8 (tvb, offset);
         offset += 1;

         if(bitone > 0x7f || (bitone > 0x0 && bitone < 0x20) ) 
         {
	    if(counter > 3)
	    {
	      if(offset > counter)
	      {
                   offset -= counter+1;

                    /* Reading the string from the packet */
                    string = tvb_get_string(tvb,offset,counter);
                    /* Printing the data */
                    proto_tree_add_string (postgresql_tree,hf_postgresql_string,tvb, offset,counter, string );
                    if (check_col (pinfo->cinfo, COL_INFO)) { col_append_fstr (pinfo->cinfo, COL_INFO, " %s", string ); }
		    g_free(string);  /* Freeing up string */
                    string=NULL;

                   offset += counter+1;
	           counter=0;
	      }
	      else
	      {
	       counter=0;
	       offset+=1;
	      }
	    }
	    else
	    {
	     counter=0;
	     offset+=1;
	    }
         }

         if( bitone == 0 )
         {
                if(counter != 0)
                { 
                  if(offset > counter)
                  {
                   offset -= counter+1;
		   if( counter > 1)
		   {
                    /* Reading the string from the packet */
		    string = tvb_get_string(tvb,offset,counter);
                    /* Printing the data */
                    proto_tree_add_string (postgresql_tree,hf_postgresql_string,tvb, offset,counter, string );
                    if (check_col (pinfo->cinfo, COL_INFO)) { col_append_fstr (pinfo->cinfo, COL_INFO, " %s", string ); }
		    g_free(string);  /* Freeing up string */
                    string=NULL;

                   }
		   offset += counter+1;
                  }
                  counter = 0;
                }
                counter=0;
         }
         else
         {
             counter += 1;
         }

         buff_remaining = tvb_length_remaining (tvb, offset);
     }
} 


void proto_register_postgresql (void)
{

  static hf_register_info hf[] = {
    {&hf_postgresql_response,
     {"Response", "postgresql.response",
      FT_BOOLEAN, BASE_NONE, NULL, 0x0,
      "TRUE if postgresql response", HFILL}},
    {&hf_postgresql_request,
     {"Request", "postgresql.request",
      FT_BOOLEAN, BASE_NONE, NULL, 0x0,
      "TRUE if postgresql request", HFILL}},
    {&hf_postgresql_string, {"String", "hf_postgresql_string", FT_STRING, BASE_NONE, NULL, 0x0, "", HFILL}},
    {&hf_postgresql_length, {"Length", "hf_postgresql_length", FT_UINT32, BASE_DEC, NULL, 0x0, "", HFILL}},
    {&hf_postgresql_string_size, {"Size", "hf_postgresql_string_size", FT_UINT32, BASE_DEC, NULL, 0x0, "", HFILL}},
    {&hf_postgresql_total_length, {"TotalLength", "hf_postgresql_total_length", FT_UINT16, BASE_DEC, NULL, 0x0, "", HFILL}},
    {&hf_postgresql_buff_remaining, {"Buffer Remaining", "hf_postgresql_buff_remaining", FT_UINT8, BASE_DEC, NULL, 0x0, "", HFILL}},
    {&hf_postgresql_opcode, {"Op Code", "hf_postgresql_opcode", FT_UINT32, BASE_HEX, NULL, 0x0, "", HFILL}},
    {&hf_postgresql_bitone, {"Bitone", "hf_postgresql_bitone", FT_UINT8, BASE_HEX, NULL, 0x0, "", HFILL}},
    {&hf_postgresql_idone, {"idone", "hf_postgresql_idone", FT_UINT8, BASE_HEX, NULL, 0x0, "", HFILL}},
 


  };

  static gint *ett[] = {
    &ett_postgresql,
  };

  proto_postgresql =
    proto_register_protocol ("POSTGRESQL", "POSTGRESQL", "postgresql");
  proto_register_field_array (proto_postgresql, hf, array_length (hf));
  proto_register_subtree_array (ett, array_length (ett));
}

void
proto_reg_handoff_postgresql (void)
{
  dissector_handle_t postgresql_handle;

  postgresql_handle =
    create_dissector_handle (dissect_postgresql, proto_postgresql);
  dissector_add ("tcp.port", TCP_PORT_POSTGRESQL, postgresql_handle);
}
