/* NoThanks.java  */

/* 
 * Copyright (C) 1996-98 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin.filter;

import muffin.*;
import muffin.html.*;
import JP.ac.osaka_u.ender.util.regex.*;

import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;
import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StreamTokenizer;

public class NoThanks implements FilterFactory
{
    FilterManager manager;
    Prefs prefs;
    NoThanksFrame frame = null;
    MessageArea messages = null;
    String hyperlist[] = { "href", "src", "action", "base", "background" };
    
    private RegExp kill = null;
    private RegExp comment = null;
    private RegExp hypertags = null;
    private RegExp content = null;
    private Hashtable strip = null;
    private Vector redirectPatterns = null;
    private Vector redirectLocations = null;
    private Hashtable replace = null;
    private Hashtable tagattrTags = null;
    private Hashtable tagattrStrip = null;
    private Hashtable tagattrRemove = null;
    private Hashtable tagattrReplace = null;
    private Hashtable tagattrReplaceValue = null;
    private StringBuffer killBuffer = null;
    private StringBuffer commentBuffer = null;
    private StringBuffer contentBuffer = null;

    public NoThanks ()
    {
	try
	{
	    hypertags = new RegExp ("^a|img|background|form|iframe|layer$");
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }

    public void setManager (FilterManager manager)
    {
	this.manager = manager;
    }
    
    public void setPrefs (Prefs prefs)
    {
	this.prefs = prefs;
	boolean o = prefs.getOverride ();
	prefs.setOverride (false);
	String filename = prefs.getUserFile ("killfile");
	prefs.putString ("NoThanks.killfile", filename);
	prefs.putInteger ("NoThanks.historySize", 500);
	prefs.setOverride (o);
	messages = new MessageArea (prefs.getInteger ("NoThanks.historySize"));
	load ();
    }

    public Prefs getPrefs ()
    {
	return prefs;
    }

    public void viewPrefs ()
    {
	if (frame == null)
	{
	    frame = new NoThanksFrame (prefs, this);
	}
	frame.setVisible (true);
    }
    
    public Filter createFilter ()
    {
	Filter f = new NoThanksFilter (this);
	f.setPrefs (prefs);
	return f;
    }

    public void shutdown ()
    {
	if (frame != null)
	{
	    frame.dispose ();
	}
    }

    boolean isKilled (String pattern)
    {
	if (kill == null)
	{
	    return false;
	}
	
	MatchInfo info = kill.match (pattern);
	return info != null;
    }

    boolean killComment (String pattern)
    {
	if (comment == null)
	{
	    return false;
	}

	MatchInfo info = comment.match (pattern);
	return info != null;
    }

    boolean killContent (String pattern)
    {
	if (content == null)
	{
	    return false;
	}

	MatchInfo info = content.match (pattern);
	return info != null;
    }

    boolean stripTag (String pattern)
    {
	return strip.containsKey (pattern);
    }

    String stripUntil (String pattern)
    {
	String s = (String) strip.get (pattern);
	return (s.length () == 0) ? null : s;
    }

    boolean replaceTag (String pattern)
    {
	return replace.containsKey (pattern);
    }

    Tag replaceTagWith (String pattern)
    {
	return (Tag) replace.get (pattern);
    }
    
    String redirect (String pattern)
    {
	for (int i = 0; i < redirectPatterns.size (); i++)
	{
	    RegExp regexp = (RegExp) redirectPatterns.elementAt (i);
	    MatchInfo info = regexp.match (pattern);
	    if (info != null)
	    {
		return (String) redirectLocations.elementAt (i);
	    }
	}
	return null;
    }

    boolean checkTag (String pattern)
    {
	MatchInfo info = hypertags.match (pattern);
	return info != null;
    }

    boolean compare (String pattern, RegExp regexp)
    {
	/* Check for empty patterns vs. ".*" */
	if (pattern.length () == 0 && regexp.toString ().equals (".*"))
	{
	    return true;
	}
	MatchInfo info = regexp.match (pattern);
	return info != null;
    }

    boolean checkTagAttributes (Tag tag)
    {
	return tagattrTags.containsKey (tag.name ());
    }
    
    boolean processTagAttributes (Tag tag)
    {
	Enumeration attrs = tag.enumerate ();
	if (attrs == null)
	{
	    return false;
	}
	
	while (attrs.hasMoreElements ())
	{
	    String name = (String) attrs.nextElement ();
	    String key = tag.name () + "." + name;
	    if (tagattrStrip.containsKey (key))
	    {
		if (compare (tag.get (name), (RegExp) tagattrStrip.get (key)))
		{
		    process ("tagattr stripped " + tag.toString () + "\n");
		    return true;
		}
	    }
	    else if (tagattrRemove.containsKey (key))
	    {
		if (compare (tag.get (name), (RegExp) tagattrRemove.get (key)))
		{
		    process ("tagattr removed " + name + " from " + tag.name () + "\n");
		    tag.remove (name);
		}
	    }
	    else if (tagattrReplace.containsKey (key))
	    {
		if (compare (tag.get (name), (RegExp) tagattrReplace.get (key)))
		{
		    process ("tagattr replaced " + name + " in " + tag.name () + "\n");
		    tag.put (name, (String) tagattrReplaceValue.get (key));
		}
	    }
	}
	return false;
    }

    void save ()
    {
	manager.save (this);
    }

    void load ()
    {
	String filename = prefs.getUserFile (prefs.getString ("NoThanks.killfile"));
	try
	{
	    //System.out.println ("NoThanks loading " + filename);
	    load (new FileReader (new File (filename)));
	}
	catch (Exception e)
	{
	}
    }

    void load (Reader reader)
    {
	strip = new Hashtable (33);
	redirectPatterns = new Vector ();
	redirectLocations = new Vector ();
	replace = new Hashtable (33);
	tagattrTags = new Hashtable (33);
	tagattrStrip = new Hashtable (33);
	tagattrRemove = new Hashtable (33);
	tagattrReplace = new Hashtable (33);
	tagattrReplaceValue = new Hashtable (33);
	killBuffer = new StringBuffer ();
	commentBuffer = new StringBuffer ();
	contentBuffer = new StringBuffer ();

	include (reader);
	
	try
	{
	    //System.out.println (killBuffer.toString ());
	    //long time = System.currentTimeMillis ();
	    kill = (killBuffer.length () > 0) ? new RegExp (killBuffer.toString ()) : null;
	    //System.out.println ("Time: " + (System.currentTimeMillis () - time));
	    comment = (commentBuffer.length () > 0) ? new RegExp (commentBuffer.toString ()) : null;
	    content = (contentBuffer.length () > 0) ? new RegExp (contentBuffer.toString ()) : null;
	}
	catch (RegExpSyntaxException e)
	{
	    System.out.println ("NoThanks RegExpSyntaxException: " + e.getMessage ());
	}
	catch (NFABuildException e)
	{
	    System.out.println ("NoThanks NFABuildException: " + e.getMessage ());
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }

    void include (Reader reader)
    {
	try
	{
	    String s;
	    int token;
	    BufferedReader in = new BufferedReader (reader);
	    while ((s = in.readLine ()) != null)
	    {
		StreamTokenizer st = new StreamTokenizer (new StringReader (s));
		st.resetSyntax ();
		st.whitespaceChars (0, 32);
		st.wordChars (33, 126);
		st.quoteChar ('"');
		st.eolIsSignificant (true);

		token = st.nextToken ();
		if (token != StreamTokenizer.TT_WORD)
		{
		    continue;
		}

		if (st.sval.startsWith ("#"))
		{
		    if (st.sval.equals ("#include"))
		    {
			token = st.nextToken ();
			if (token != StreamTokenizer.TT_WORD && token != '"')
			{
			    break;
			}
			String filename = prefs.getUserFile (st.sval);
			include (new FileReader (new File (filename)));
		    }
		    continue;
		}
		
		if (st.sval.equals ("kill"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    if (killBuffer.length () > 0)
		    {
			killBuffer.append ("|");
		    }
		    killBuffer.append (st.sval);
		}
		else if (st.sval.equals ("comment"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    if (commentBuffer.length () > 0)
		    {
			commentBuffer.append ("|");
		    }
		    commentBuffer.append (st.sval);
		}
		else if (st.sval.equals ("strip"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    String start = new String (st.sval);
		    String end = "";
		    token = st.nextToken ();
		    if (token == StreamTokenizer.TT_WORD || token == '"')
		    {
			end = new String (st.sval);
		    }
		    strip.put (start.toLowerCase (), end.toLowerCase ());
		}
		else if (st.sval.equals ("content"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    if (contentBuffer.length () > 0)
		    {
			contentBuffer.append ("|");
		    }
		    contentBuffer.append (st.sval); 		
		}
		else if (st.sval.equals ("redirect"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    String pattern = new String (st.sval);
		    String location = "";
		    token = st.nextToken ();
		    if (token == StreamTokenizer.TT_WORD || token == '"')
		    {
			location = new String (st.sval);
		    }
		    try
		    {
			RegExp regexp = new RegExp (pattern);
			redirectPatterns.addElement (regexp);
			redirectLocations.addElement (location);
		    }
		    catch (RegExpSyntaxException e)
		    {
			System.out.println (pattern + " " + e.getMessage ());
		    }
		    catch (NFABuildException e)
		    {
			System.out.println (pattern + " " + e.getMessage ());
		    }
		}
		else if (st.sval.equals ("replace"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    String oldtag = new String (st.sval);
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    String newtag = new String (st.sval);
		    String name = null;
		    String data = null;
		    int i = newtag.indexOf (" \t");
		    if (i == -1)
		    {
			name = newtag;
		    }
		    else
		    {
			name = newtag.substring (i);
			data = newtag.substring (i+1);
		    }
		    Tag tag = new Tag (name, data);
		    replace.put (oldtag.toLowerCase (), tag);
		}
		else if (st.sval.equals ("tagattr"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    int i = st.sval.indexOf ('.');
		    if (i == -1)
		    {
			break;
		    }
		    String tag = st.sval.substring (0, i);
		    tag = tag.toLowerCase ();
		    String attr = st.sval.substring (i+1);
		    attr = attr.toLowerCase ();
		    String key = tag + "." + attr;
		    
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    String command = new String (st.sval);

		    Vector list = (Vector) tagattrTags.get (tag);
		    if (list == null)
		    {
			list = new Vector ();
			tagattrTags.put (tag, list);
		    }
		    list.addElement (attr);

		    String pattern;
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			pattern = ".*";
		    }
		    else
		    {
			pattern = new String (st.sval);
		    }

		    RegExp regex = new RegExp (pattern);
		
		    if (command.equals ("strip"))
		    {
			tagattrStrip.put (key, regex);
		    }
		    else if (command.equals ("remove"))
		    {
			tagattrRemove.put (key, regex);
		    }
		    else if (command.equals ("replace"))
		    {
			token = st.nextToken ();
			if (token != StreamTokenizer.TT_WORD && token != '"')
			{
			    System.out.println ("tagattr replace missing value");
			    break;
			}
			String value = new String (st.sval);

			tagattrReplace.put (key, regex);
			tagattrReplaceValue.put (key, value);
		    }
		    else
		    {
			System.out.println ("tagattr " + command + " unknown command");
		    }
		}
		else
		{
		    System.out.println ("NoThanks: " + st.sval + " unknown command");
		}
	    }
 	    in.close ();
	}
	catch (RegExpSyntaxException e)
	{
	    System.out.println ("NoThanks RegExpSyntaxException: " + e.getMessage ());
	}
	catch (NFABuildException e)
	{
	    System.out.println ("NoThanks NFABuildException: " + e.getMessage ());
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }

    void process (String s)
    {
	messages.append (s);
    }
}

