/* Tag.java */

/* 
 * Copyright (C) 1997 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin.html;

import java.util.Hashtable;
import java.util.Enumeration;
import sdsu.util.TokenCharacters;
import sdsu.util.SimpleTokenizer;

public class Tag
{
    protected String name = "";
    protected byte contents[] = null;
    protected int contentsIndex = -1;
    protected int contentsEnd = -1;

    private Hashtable attributes = null;
    private boolean parsed = false;
    private boolean modified = false;

    private void parse ()
    {
	parsed = true;

	if (contentsEnd - contentsIndex <= 0
	    || name.length () <= 0
	    || name.startsWith ("<!doctype"))
	{
	    return;
	}
	
	try
	{
	    String str = new String (contents, contentsIndex, contentsEnd - contentsIndex);
	    
	    TokenCharacters chars = new TokenCharacters ("", "", '"', '"', " \t\r\n");
	    chars.addQuoteChars ('\'', '\'');

	    SimpleTokenizer st = new SimpleTokenizer (str, chars);
	    st.setEatEscapeChar (false);

	    while (st.hasMoreTokens ())
	    {
		String key = st.nextToken ("=");
		String value;
		if (st.separator () == '=')
		{
		    value = st.nextToken ("");
		}
		else
		{
		    value = "";
		}
		//System.out.println ("key="+key+" value="+value);
		put (key, value);
	    }
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }

    public String name ()
    {
	return name;
    }

    public boolean is (String s)
    {
	return name.equals (s);
    }

    public boolean has (String key)
    {
	if (!parsed) parse ();
	return attributes != null ? attributes.containsKey (key) : false;
    }

    public String get (String key)
    {
	if (!parsed) parse ();
	return attributes != null ? (String) attributes.get (key) : null;
    }

    public void put (String key, String value)
    {
	if (!parsed) parse ();
	if (attributes == null)
	{
	    attributes = new Hashtable (13);
	}
	attributes.put (key.toLowerCase (), value);
	modified = true;
    }

    public void remove (String key)
    {
	if (!parsed) parse ();
	if (attributes != null)
	{
	    attributes.remove (key);
	    modified = true;
	}
    }

    public void rename (String newName)
    {
	name = newName;
	modified = true;
    }

    public boolean isModified ()
    {
	return modified;
    }
    
    public String toString ()
    {
	if (!parsed) parse ();

	StringBuffer buf = new StringBuffer ();
	buf.append ("<");
	buf.append (name);
	if (attributes != null && !attributes.isEmpty ())
	{
	    String key, value;
	    Enumeration e = attributes.keys ();
	    while (e.hasMoreElements ())
	    {
		key = (String) e.nextElement ();
		buf.append (" ");
		buf.append (key);
		value = (String) get (key);
		if (value.length () > 0)
		{
		    buf.append ("=");
		    buf.append ("\"");
		    buf.append (value);
		    buf.append ("\"");
		}
	    }
	}
	buf.append (">");
	return buf.toString ();
    }
}
