/* NoThanks.java  */

/* 
 * Copyright (C) 1997 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin.filter;

import muffin.*;
import muffin.html.*;
import JP.ac.osaka_u.ender.util.regex.*;

import java.util.Hashtable;
import java.util.Vector;
import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.StringReader;
import java.io.StreamTokenizer;

public class NoThanks implements FilterFactory
{
    FilterManager manager;
    Prefs prefs;
    NoThanksFrame frame = null;
    String hyperlist[] = { "href", "src", "action", "base", "background" };
    
    private RegExp kill = null;
    private RegExp comment = null;
    private RegExp hypertags = null;
    private RegExp content = null;
    private Hashtable strip = null;
    private Vector redirectPatterns = null;
    private Vector redirectLocations = null;

    public NoThanks ()
    {
	try
	{
	    hypertags = new RegExp ("^a|img|background$");
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }

    public void setManager (FilterManager manager)
    {
	this.manager = manager;
    }
    
    public void setPrefs (Prefs prefs)
    {
	this.prefs = prefs;
	boolean o = prefs.getOverride ();
	prefs.setOverride (false);
	String filename = prefs.getUserFile ("killfile");
	prefs.putString ("NoThanks.killfile", filename);
	prefs.setOverride (o);
	load ();
    }

    public Prefs getPrefs ()
    {
	return prefs;
    }

    public void viewPrefs ()
    {
	if (frame == null)
	{
	    frame = new NoThanksFrame (prefs, this);
	}
	frame.setVisible (true);
    }
    
    public Filter createFilter ()
    {
	Filter f = new NoThanksFilter (this);
	f.setPrefs (prefs);
	return f;
    }

    boolean isKilled (String pattern)
    {
	if (kill == null)
	{
	    return false;
	}
	
	MatchInfo info = kill.match (pattern);
	return info != null;
    }

    boolean killComment (String pattern)
    {
	if (comment == null)
	{
	    return false;
	}

	MatchInfo info = comment.match (pattern);
	return info != null;
    }

    boolean killContent (String pattern)
    {
	if (content == null)
	{
	    return false;
	}

	MatchInfo info = content.match (pattern);
	return info != null;
    }

    boolean stripTag (String pattern)
    {
	return strip.containsKey (pattern);
    }

    String stripUntil (String pattern)
    {
	String s = (String) strip.get (pattern);
	return (s.length () == 0) ? null : s;
    }

    String redirect (String pattern)
    {
	for (int i = 0; i < redirectPatterns.size (); i++)
	{
	    RegExp regexp = (RegExp) redirectPatterns.elementAt (i);
	    MatchInfo info = regexp.match (pattern);
	    if (info != null)
	    {
		return (String) redirectLocations.elementAt (i);
	    }
	}
	return null;
    }

    boolean checkTag (String pattern)
    {
	MatchInfo info = hypertags.match (pattern);
	return info != null;
    }

    void save ()
    {
	manager.save (this);
    }

    void load ()
    {
	strip = new Hashtable (33);

	redirectPatterns = new Vector ();
	redirectLocations = new Vector ();
	
	String filename = prefs.getString ("NoThanks.killfile");
	System.out.println ("NoThanks loading " + filename);
	File file = new File (filename);
	if (!file.exists ())
	{
	    System.out.println ("NoThanks can't open " + filename);
	    return;
	}
	try
	{
 	    BufferedReader in = new BufferedReader (new FileReader (file));
	    String s;
	    int token;
	    StringBuffer killBuffer = new StringBuffer ();
	    StringBuffer commentBuffer = new StringBuffer ();
	    StringBuffer contentBuffer = new StringBuffer ();
	    while ((s = in.readLine ()) != null)
	    {
		if (s.startsWith ("#"))
		{
		    continue;
		}
		
		StreamTokenizer st = new StreamTokenizer (new StringReader (s));
		st.resetSyntax ();
		st.commentChar ('#');
		st.whitespaceChars (0, 32);
		st.wordChars (33, 126);
		st.quoteChar ('"');
		st.eolIsSignificant (true);

		token = st.nextToken ();
		if (token != StreamTokenizer.TT_WORD)
		{
		    continue;
		}
		
		if (st.sval.equals ("kill"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    if (killBuffer.length () > 0)
		    {
			killBuffer.append ("|");
		    }
		    killBuffer.append (st.sval);
		}
		else if (st.sval.equals ("comment"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    if (commentBuffer.length () > 0)
		    {
			commentBuffer.append ("|");
		    }
		    commentBuffer.append (st.sval);
		}
		else if (st.sval.equals ("strip"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    String start = new String (st.sval);
		    String end = "";
		    token = st.nextToken ();
		    if (token == StreamTokenizer.TT_WORD || token == '"')
		    {
			end = new String (st.sval);
		    }
		    strip.put (start, end);
		}
		else if (st.sval.equals ("content"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    if (contentBuffer.length () > 0)
		    {
			contentBuffer.append ("|");
		    }
		    contentBuffer.append (st.sval); 		
		}
		else if (st.sval.equals ("redirect"))
		{
		    token = st.nextToken ();
		    if (token != StreamTokenizer.TT_WORD && token != '"')
		    {
			break;
		    }
		    String pattern = new String (st.sval);
		    String location = "";
		    token = st.nextToken ();
		    if (token == StreamTokenizer.TT_WORD || token == '"')
		    {
			location = new String (st.sval);
		    }
		    try
		    {
			RegExp regexp = new RegExp (pattern);
			redirectPatterns.addElement (regexp);
			redirectLocations.addElement (location);
		    }
		    catch (RegExpSyntaxException e)
		    {
			System.out.println (pattern + " " + e.getMessage ());
		    }
		    catch (NFABuildException e)
		    {
			System.out.println (pattern + " " + e.getMessage ());
		    }
		}
		else
		{
		    System.out.println ("NoThanks: " + st.sval + " unknown command");
		}
	    }
 	    in.close ();

	    if (killBuffer.length () > 0)
	    {
		kill = new RegExp (killBuffer.toString ());
	    }
	    else
	    {
		kill = null;
	    }

	    if (commentBuffer.length () > 0)
	    {
		comment = new RegExp (commentBuffer.toString ());
	    }
	    else
	    {
		comment = null;
	    }

	    if (contentBuffer.length () > 0)
	    {
		content = new RegExp (contentBuffer.toString ());
	    }
	    else
	    {
		content = null;
	    }
	}
	catch (RegExpSyntaxException e)
	{
	    System.out.println ("NoThanks RegExpSyntaxException: " + e.getMessage ());
	}
	catch (NFABuildException e)
	{
	    System.out.println ("NoThanks NFABuildException: " + e.getMessage ());
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }

    void process (String s)
    {
	if (frame != null)
	{
	    frame.text.append (s);
	}
    }
}

