/* Handler.java */

/* 
 * Copyright (C) 1996-97 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.Socket;
import java.net.URL;

/**
 * HTTP transaction handler.  A handler is created by muffin.Server for
 * each HTTP transaction.  Given a socket, the handler will deal with
 * the request, reply, and invoke request, reply, and content filters.
 *
 * @see muffin.Server
 * @author Mark Boyns
 */
public class Handler extends Thread
{
    Monitor monitor = null;
    FilterManager manager = null;
    Options options = null;
    Client client = null;
    Socket socket = null;
    Request request = null;
    Reply reply = null;
    HttpRelay http = null;
    int currentLength = 0;
    int contentLength = 0;
    Filter filterList[];
    boolean isFiltered = false;

    /**
     * Create a Handler.
     */
    public Handler (ThreadGroup t, Runnable r, Monitor m, FilterManager manager, Options options)
    {
	super (t, r);
	this.monitor = m;
	this.manager = manager;
	this.options = options;
    }

    /**
     * Start the handler.
     *
     * @param s a socket
     */
    public void doit (Socket s)
    {
	socket = s;
	start ();
    }

    /**
     * Close all connections associated with this handler.
     */
    public synchronized void close ()
    {
	if (client != null)
	{
	    client.close ();
	}
	if (http != null)
	{
	    http.close ();
	}
    }

    /**
     * Where all the work gets done.
     */
    public void run ()
    {
	filterList = manager.createFilters ();
	
	try
	{
	    client = new Client (socket);
	    request = client.read ();
	    
	    if (!request.getURL ().startsWith ("http://"))
	    {
		request.setURL ("http://" + options.getString ("muffin.host")
				+ ":" + options.getString ("muffin.port")
				+ request.getURL ());
	    }
	    
	    if (!options.getBoolean ("muffin.passthru"))
	    {
		filter (request);
	    }
	    monitor.register (this);

	    if (Httpd.sendme (request))
	    {
		http = new Httpd (socket);
	    }
	    else if (options.useProxy ())
	    {
		http = new Proxy (options.getString ("muffin.httpProxyHost"),
				  options.getInteger ("muffin.httpProxyPort"));
	    }
	    else
	    {
		http = new Http (new URL (request.getURL ()));
	    }
	    http.write (request);
	    
	    monitor.update (this);

	    reply = http.read ();
	    if (!options.getBoolean ("muffin.passthru"))
	    {
		filter (reply);
	    }
	    monitor.update (this);

	    try 
	    {
		contentLength = Integer.parseInt (reply.getHeaderField ("Content-length"));
	    }
	    catch (Exception e)
	    {
		contentLength = 0;
	    }
	    currentLength = 0;
	    monitor.update (this);

	    //long start = System.currentTimeMillis ();
	    
	    if (!options.getBoolean ("muffin.passthru"))
	    {
		filter (http.getInputStream (), client.getOutputStream ());
	    }
	    else
	    {
		client.write (reply);
		copy (http.getInputStream (), client.getOutputStream (), true);
	    }
	    
 	    //long end = System.currentTimeMillis ();
 	    //long elapsed = end - start;
 	    //System.out.println ("time " + elapsed);
	}
	catch (Exception e)
	{
	    error (client.getOutputStream (), e, request);
	}
	
	close ();
	
	monitor.unregister (this);
    }

    /**
     * Run a reply through all the filters.
     *
     * @param r a reply
     */
    void filter (Reply r) throws FilterException
    {
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof ReplyFilter)
	    {
		((ReplyFilter)(filterList[i])).filter (r);
	    }
	}
    }

    /**
     * Run a request through all the filters.
     *
     * @param r a request
     */
    void filter (Request r) throws FilterException
    {
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof RequestFilter)
	    {
		((RequestFilter)(filterList[i])).filter (r);
	    }
	}
    }

    void xfilter (InputStream in, OutputStream out) throws java.io.IOException
    {
	int bufsize = contentLength > 0 ? contentLength : 8192;
	ByteArrayInputStream inbuf;
	ByteArrayOutputStream outbuf;

	outbuf = new ByteArrayOutputStream (bufsize);
	System.out.println ("Reading into memory");
	copy (in, outbuf, true);
	System.out.println (outbuf.size () + " bytes");

	System.out.println ("Filtering");
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof ContentFilter)
	    {
		ContentFilter filter = (ContentFilter) filterList[i];
		if (filter.needsFiltration (request, reply))
		{
		    inbuf = new ByteArrayInputStream (outbuf.toByteArray ());
		    outbuf = new ByteArrayOutputStream (bufsize);
		    
		    filter.setInputStream (inbuf);
		    filter.setOutputStream (outbuf);
		    filter.run ();
		}
	    }
	}
	
	client.write (reply);
	System.out.println ("Sending to client");
	copy (new ByteArrayInputStream (outbuf.toByteArray ()), out, false);
    }
    
    /**
     * Run the content through all the filters.
     *
     * @param in a reader
     * @param out a writer
     */
    void filter (InputStream in, OutputStream out) throws java.io.IOException
    {
	int pipeSize = 8192;
	
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof ContentFilter)
	    {
		ContentFilter filter = (ContentFilter) filterList[i];
		if (filter.needsFiltration (request, reply))
		{
		    PipedOutputStream po = new PipedOutputStream ();
		    AdjustablePipedInputStream pi = new AdjustablePipedInputStream (po, pipeSize);

		    filter.setInputStream (in);
		    filter.setOutputStream (po);
		
		    Thread t = new Thread (filter);
		    try
		    {
			t.setPriority (Thread.MIN_PRIORITY);
		    }
		    catch (Exception e)
		    {
			e.printStackTrace ();
		    }
		    t.start ();

		    in = pi;
		    isFiltered = true;
		}
	    }
	}

	if (isFiltered)
	{
	    int bufsize = contentLength > 0 ? contentLength : pipeSize;
	    ByteArrayOutputStream outbuf = new ByteArrayOutputStream (bufsize);
	    copy (in, outbuf, true);
	    ByteArrayInputStream inbuf = new ByteArrayInputStream (outbuf.toByteArray ());
	    if (reply.containsHeaderField ("Content-length"))
	    {
		reply.setHeaderField ("Content-length", outbuf.size ());
	    }
	    client.write (reply);
	    copy (inbuf, out, false);
	}
	else
	{
	    client.write (reply);
	    copy (in, out, true);
	}
    }

    /**
     * Return the content length.
     */
    public int getTotalBytes ()
    {
	return contentLength > 0 ? contentLength : 0;
    }

    /**
     * Return the number of bytes read so far.
     */
    public int getCurrentBytes ()
    {
	return currentLength;
    }

    /**
     * Return a string represenation of the hander's state.
     */
    public String toString ()
    {
	StringBuffer str = new StringBuffer ();
	if (contentLength > 0)
	{
	    str.append ((int)(((float)currentLength/contentLength)*100));
	    str.append ("% of ");
	    str.append (contentLength/1024);
	    str.append ("k");
	}
	else
	{
	    str.append (currentLength/1024);
	    str.append ("k");
	}
	str.append (" ");
	str.append (request.getURL ());
	
	return str.toString ();
    }

    /**
     * Send a error message to the client.
     *
     * @param out client
     * @param e exception that occurred
     * @param r request
     */
    void error (OutputStream out, Exception e, Request r)
    {
	//e.printStackTrace ();
	
	StringBuffer buf = new StringBuffer ();
	buf.append ("While trying to retrieve the URL: <a href=\""+r.getURL()+"\">"+r.getURL()+"</a>\r\n");
	buf.append ("<p>\r\nThe following error was encountered:\r\n<p>\r\n");
	buf.append ("<ul><li>" + e.toString () + "</ul>\r\n");
	String s = (new HttpError (options, 400, buf.toString ())).toString ();
	try
	{
	    out.write (s.getBytes (), 0, s.length ());
	    out.flush ();
	}
	catch (Exception ex)
	{
	}
    }

    /**
     * Copy input reader to the output writer.
     *
     * @param in reader
     * @param out writer
     */
    void copy (InputStream in, OutputStream out, boolean monitored) throws java.io.IOException
    {
	int n;
	int copySize = 8192;
	byte buffer[] = new byte[copySize];

	while ((n = in.read (buffer, 0, copySize)) > 0)
	{
	    out.write (buffer, 0, n);
	    if (monitored)
	    {
		//System.out.println ("read " + n);
		currentLength += n;
		monitor.update (this);
	    }
	}
	out.flush ();
    }
}
