/* FilterManager.java */

/* 
 * Copyright (C) 1997 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.util.Vector;

/**
 * Class to manage all filters.  This class maintains the list of
 * known filters, enabled filters, and all filter preferences.
 *
 * @see muffin.Filter
 * @see muffin.FilterFactory
 * @author Mark Boyns
 */
public class FilterManager
{
    Vector knownFilters = null;
    Vector enabledFilters = null;
    UserPrefs userPrefs;
    FilterManagerFrame frame = null;

    /**
     * Create a FilterManager.
     *
     * @param userPrefs user preferences
     */
    public FilterManager (UserPrefs userPrefs)
    {
	final String knownList[] = 
	{ 
	    "muffin.filter.AnimationKiller",
	    "muffin.filter.CookieMonster",
	    "muffin.filter.Decaf",
	    "muffin.filter.DocumentInfo",
	    "muffin.filter.Glossary",
	    "muffin.filter.Logger",
	    "muffin.filter.Painter",
	    "muffin.filter.Preview",
	    "muffin.filter.NoThanks",
	    "muffin.filter.Referer",
	    "muffin.filter.Secretary",
	    "muffin.filter.SecretAgent",
	    "muffin.filter.SecretServer",
	    "muffin.filter.Snoop",
	    "muffin.filter.Stats",
	};
	
	this.userPrefs = userPrefs;
	
	int i;
	String list[];

	/* known filters */
	knownFilters = new Vector (32);
	list = userPrefs.getStringList ("muffin.knownFilters");
	if (list.length == 0)
	{
	    list = knownList;
	}
	for (i = 0; i < list.length; i++)
	{
	    knownFilters.addElement (list[i]);
	}

	/* enabled filters */
	enabledFilters = new Vector (32);
	list = userPrefs.getStringList ("muffin.enabledFilters");
	for (i = 0; i < list.length; i++)
	{
	    enable (list[i]);
	}
    }

    /**
     * Return a list of filters created by each filter's
     * factory method.
     *
     * @see muffin.Handler
     */
    public Filter[] createFilters ()
    {
	FilterFactory ff;
	Filter list[] = new Filter[enabledFilters.size ()];
	for (int i = 0; i < list.length; i++)
	{
	    ff = (FilterFactory) enabledFilters.elementAt (i);
	    list[i] = ff.createFilter ();
	}
	return list;
    }

    public void createFrame ()
    {
	if (frame == null)
	{
	    frame = new FilterManagerFrame (this);
	}
	frame.hideshow ();
    }

    public void append (String clazz)
    {
	knownFilters.addElement (clazz);
	if (frame != null)
	{
	    frame.updateKnownFiltersList ();
	}
    }

    public void remove (int i)
    {
	knownFilters.removeElementAt (i);
	if (frame != null)
	{
	    frame.updateKnownFiltersList ();
	}
    }

    /**
     * Enable a filter
     *
     * @param clazz java class name
     */
    public void enable (String clazz)
    {
	try
	{
	    System.out.println ("Enabling " + clazz);
	    FilterFactory ff = (FilterFactory) (Class.forName (clazz)).newInstance ();
	    Prefs prefs = userPrefs.extract (clazz.substring (clazz.lastIndexOf ('.') + 1));
	    ff.setPrefs (prefs);
	    ff.setManager (this);
	    enabledFilters.addElement (ff);

	    // Update muffin.enabledFilters
	    String list[] = new String[enabledFilters.size ()];
	    for (int i = 0; i < list.length; i++)
	    {
		ff = (FilterFactory) enabledFilters.elementAt (i);
		list[i] = (ff.getClass ()).getName ();
	    }
	    userPrefs.putStringList ("muffin.enabledFilters", list);

	    if (frame != null)
	    {
		frame.updateEnabledFiltersList ();
	    }
	}
	catch (Exception e)
	{
	    System.out.println ("");
	    System.out.println ("WARNING: Can't load " + clazz + ": ");
	    System.out.println ("");
	    System.out.println ("         " + e);
	    System.out.println ("");
	    System.out.println ("         You may need to restart muffin with a different CLASSPATH.");
	    System.out.println ("");
	}
    }

    public void disable (int i)
    {
	FilterFactory ff = (FilterFactory) enabledFilters.elementAt (i);
	System.out.println ("Disabling " + ff.getClass ().getName ());
	enabledFilters.removeElementAt (i);

	if (frame != null)
	{
	    frame.updateEnabledFiltersList ();
	}
    }

    /**
     * Save all filter preferences.
     */
    public void save ()
    {
	userPrefs.save ();
    }

    /**
     * Save filter preferences.
     *
     * @param ff filter factory
     */
    public void save (FilterFactory ff)
    {
	userPrefs.merge (ff.getPrefs ());
	userPrefs.save ();
    }
}    
