/* ssmap.c
 * Copyright (C) 1994 Peter Ross
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License, see the file COPYING.
 *
 * Outputs a map of a spatially structured population, for later analysis.
 * Written by Peter Ross.
 */

#include <stdio.h>
#include "pga.h"

#define TRUE   1
#define FALSE  0


extern int numchromo;
extern int ss_xmax, ss_ymax;
extern char *malloc();
extern int geno_size;
extern void qsort();

struct CHROMOTYPE {   /* cell for recording the type of a chromo */
   struct CHROMOTYPE *next;   /* chain pointer */
   int typeval;               /* unique ID for this chromo string */
   int instances;             /* how many times it appears */
   char typechar;             /* what char it will be printed as */
   CHROMOSOME *cptr;          /* pointer to actual chromosome */
};

void do_one_pop();

void ss_map(fname,gen, pops, npops, popsize)
char *fname;
int gen;
CHROMOSOME *pops;
int npops, popsize;
{
  FILE *f;
  int i;
  char mapname[256];

  sprintf(mapname, "%s.map", fname);
  if((f = fopen(mapname, "a")) != (FILE *)NULL) {
    fprintf(f,"----------------------\ngeneration = %d\n", gen);
    for(i=0;i<npops;i++) {
      fprintf(f,"population = %d\n", i);
      do_one_pop(f,gen,pops,i,popsize);
    }
    fclose(f);
  }
  else
    fprintf(stderr, "*** Could not open map file ***\n");
}

int compare_by_fitness(ct1,ct2)
struct CHROMOTYPE *ct1, *ct2;
{ if(ct2->cptr->fitness < ct1->cptr->fitness)
   return(-1);
  else if(ct2->cptr->fitness == ct1->cptr->fitness)
   return(0);
  else
   return(1);
}

int compare_by_typeval(ct1,ct2)
struct CHROMOTYPE *ct1, *ct2;
{ return(ct1->typeval - ct2->typeval);
}

#define LOCATION(x,y) (thispop*range+(y)*ss_xmax+(x))

void do_one_pop(f, g, pops, thispop, range)
FILE *f;
int g;
CHROMOSOME *pops;
int thispop,range;
{
  struct CHROMOTYPE *chromodict, *tmp, *sortdict;
  CHROMOSOME *c;
  int *chromotypes;
  int curtype = 0;
  int i,j,found;
  chromodict = (struct CHROMOTYPE *)NULL;

  chromotypes = (int *)malloc(numchromo*sizeof(int));

  for(i=0; i<numchromo; i++) {
    c = &pops[thispop*range + i];
    tmp = chromodict;
    found = FALSE;
    while(tmp != (struct CHROMOTYPE *)NULL && !found) {
       /* Have we found this string already? */
       if(strncmp(tmp->cptr->gdp->genotype,c->gdp->genotype,geno_size) ==0) {
          chromotypes[i] = tmp->typeval;
          tmp->instances++;
          found = TRUE;
       }
       else
          tmp = tmp->next;
    }
    /* If we haven't seen it already .. */
    if(!found /*tmp == (struct CHROMOTYPE *)NULL*/) {
      tmp = (struct CHROMOTYPE *)malloc(sizeof(struct CHROMOTYPE));
      tmp->typeval = curtype;
      tmp->instances = 1;
      tmp->cptr = c;
      tmp->next = chromodict;
      chromodict = tmp;
      chromotypes[i] = curtype++;
    }
  }

  /* Sort out how to represent this in printout .. crude but never mind */

  /* Step 1 is to transer the linked list to an array: */
  sortdict = (struct CHROMOTYPE *)malloc(curtype*sizeof(struct CHROMOTYPE));
  for(i=curtype-1, tmp=chromodict;
      tmp != (struct CHROMOTYPE *)NULL;
      i--, tmp=tmp->next) {
    sortdict[i] = *tmp;
  }

  /* This array needs to be sorted, so max fitness comes first: */
  qsort(sortdict,curtype,(sizeof(struct CHROMOTYPE)),compare_by_fitness);

  /* Now assign characters, starting from 'A', up to 'Z' at most. */
  /* The 27th and lower fitness chromosomes appear as '.'         */
  for(i=0;i<curtype;i++)
    sortdict[i].typechar = (i<26)?'A'+i:'.';

  /* Now sort that array by typeval again, so we can index into it */
  /* by the entries in the chromotypes array                       */
  qsort(sortdict,curtype,(sizeof(struct CHROMOTYPE)),compare_by_typeval);

  /* Print the rectangular map to file */
  for(i=0; i < ss_ymax; i++) {
    for(j=0; j < ss_xmax; j++) {
      fprintf(f, " %c", sortdict[chromotypes[i*ss_xmax+j]].typechar);
    }
    fprintf(f,"\n");
  }
  fprintf(f,"\n");

  /* Print the dictionary for this map */
  /* Print it fittest first; so must sort again */
  qsort(sortdict,curtype,(sizeof(struct CHROMOTYPE)),compare_by_fitness);
  for(i=0;i<curtype && i < 26;i++) {
    fprintf(f, "%d %c: %4d %14.9f %s\n", 
               g,
               sortdict[i].typechar,
               sortdict[i].instances,
               sortdict[i].cptr->fitness,
               sortdict[i].cptr->gdp->genotype);
  }
  fprintf(f,"total no of distinct chromosomes = %d\n\n",curtype);

  /* Free up everything used this time */
  tmp = chromodict;
  while(tmp != (struct CHROMOTYPE *)NULL) {
    chromodict = tmp->next;
    free(tmp);
    tmp = chromodict;
  }
  free(chromotypes);
  free(sortdict);
}
