/* mutate.c
 * Copyright (C) 1993, 1994 Peter Ross and Geoff Ballinger.
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License, see the file COPYING.
 *
 * Mutation. mutate() takes a CHROMOSOME as argument, returns one.
 * The function mutate() points at the appropriate mutation procedure
 * in this file.
 */

#include <stdio.h>
#include "pga.h"

#if hpux || SOLARIS
#define random() lrand48()
#define srandom(seed) srand48((long) seed)
#endif hpux || SOLARIS

extern int geno_size;
extern int chromo_id;
extern int max_allele;
extern double mute_rate;
extern double drandom();
extern long random();

extern int tt_maxslots;
extern int tt_days;
extern int tt_slots_per_day;
extern int tt_spread;
extern int tt_clash_type;
extern int tt_mtn1;
extern int tt_mtn2;
extern double tt_penalty_clash;
extern double tt_penalty_order;
extern double tt_penalty_exclude;
extern double tt_penalty_consec;
extern struct constraint **tt_event_data;
extern double *tt_scores;
extern int (*find_bad_event)();
extern int (*find_good_slot)();

extern char *malloc();
extern char *strncpy();

/************************************************************************/
/* This flips a random 0/1 bit in 'original'.                               */
/* If given chromo has non-zero refcount, then work on copy instead.    */
/* Refcounts get adjusted upon insertion, later.                        */
/************************************************************************/

CHROMOSOME mutate_01(original, rate)
CHROMOSOME original;
double rate;
{
  int place;
  CHROMOSOME newchromo;

  if(original.gdp->refcount > 0) {
     newchromo.gdp = (GENODATA *)malloc(sizeof(GENODATA));
     newchromo.gdp->refcount = 0;
     newchromo.gdp->genotype = (char *)malloc(geno_size+1);
     newchromo.gdp->genotype[geno_size]='\0';
     newchromo.gdp->id = chromo_id++;
     newchromo.gdp->parent1 = original.gdp->parent1;
     newchromo.gdp->parent2 = original.gdp->parent2;
     strncpy(newchromo.gdp->genotype,original.gdp->genotype,geno_size);
  } else
     newchromo = original;

  for(place=0;place < geno_size; place++) {
     if(drandom() <= rate)
        if(newchromo.gdp->genotype[place] == '0')
          newchromo.gdp->genotype[place] = '1';
        else
          newchromo.gdp->genotype[place] = '0';
  }
  
  return(newchromo);
}

/************************************************************************/
/* This changes a random allele in the original, to lie in range '0' to */
/* '0'+max_allele. Forces a different value to to the current one.      */
/* If given chromo has non-zero refcount, then work on copy instead.    */
/* Refcounts get adjusted upon insertion, later.                        */
/************************************************************************/

CHROMOSOME mutate_multi(original, rate)
CHROMOSOME original;
double rate;
{
  int place;
  char c;
  CHROMOSOME newchromo;

  if(original.gdp->refcount > 0) {
     newchromo.gdp = (GENODATA *)malloc(sizeof(GENODATA));
     newchromo.gdp->refcount = 0;
     newchromo.gdp->genotype = (char *)malloc(geno_size+1);
     newchromo.gdp->genotype[geno_size]='\0';
     newchromo.gdp->id = chromo_id++;
     newchromo.gdp->parent1 = original.gdp->parent1;
     newchromo.gdp->parent2 = original.gdp->parent2;
     strncpy(newchromo.gdp->genotype,original.gdp->genotype,geno_size);
  } else
     newchromo = original;

  for(place=0;place < geno_size; place++) {
     if(drandom() <= rate) {
       c = '0' + (random() % max_allele);
       while(c == newchromo.gdp->genotype[place])
         c = '0' + (random() % max_allele);
       newchromo.gdp->genotype[place] = c;
     }
  }

  return(newchromo);
}

/************************************************************************/
/* Function mrate() points at one of the next two functions, depending  */
/* on whether adaptive is chosen or note. Adaptive mutation scales the  */
/* probability of mutating each bit by the degree of similarity of the  */
/* parents.                                                             */
/************************************************************************/

/************************************************************************/
/* This function returns the bit mutation rate appropriate to the given */
/* parents - a number in the range 0..mute_rate, depending on degree of */
/* similarity of parents.                                               */
/************************************************************************/

double adaptive(parent1,parent2)
CHROMOSOME parent1,parent2;
{
  int loop,count;
  double prob;

  count=0;
  for(loop=0; loop<geno_size; loop+=1) {
    if (parent1.gdp->genotype[loop] == parent2.gdp->genotype[loop]) count++;
  }
                        /* Proportion = observed / total. */
  prob=((double)count) / ((double)geno_size);

  return(prob*mute_rate);
}


/************************************************************************/
/* This function simply returns the fixed mutation rate.                */
/************************************************************************/

double nonadaptive(parent1,parent2)
CHROMOSOME parent1,parent2;
{
  return(mute_rate);
}


/***********************************************************************/
/* Timetabling: event-freeing mutation. It finds events which are      */
/* heavily contributing to the penalty, and picks one. It changes the  */
/* slot of that event to one which cuts down the contribution a lot.   */
/***********************************************************************/

/* Work out the violation score for a given event assuming it is in a */
/* given slot in the given genotype.                                  */

double violation_score(event,slot,genotype)
int event, slot;
char *genotype;
{
  double spenalty;
  int otherslot;
  struct constraint *ctmp;

  spenalty = 0.0;
  for(ctmp = tt_event_data[event];
      ctmp != (struct constraint *)NULL;
      ctmp = ctmp->next) {
    switch(ctmp->ctype) {
      case CLASH:
      case CLASH_COPY:
        otherslot = SLOTVAL(genotype[ctmp->e2]);
        if(slot == otherslot)
          spenalty += tt_penalty_clash;
        else if(abs(slot - otherslot) <= tt_spread
                && (slot/tt_slots_per_day) == (otherslot/tt_slots_per_day))
          spenalty += (tt_penalty_consec/(double)abs(slot-otherslot));
        break;
      case BEFORE:
        otherslot = SLOTVAL(genotype[ctmp->e2]);
        if(slot >= otherslot)
          spenalty += tt_penalty_order;
        break;
      case EXCLUDE:
        if(slot == ctmp->s)
          spenalty += tt_penalty_exclude;
        break;
      default:
        fprintf(stderr,"PGA: unknown constraint type found when mutating\n");
        exit(1);
        break;
    } /* end of switch */
  } /* end of adding up penalties for this event */
  return(spenalty);
}

/*********************************************************************/ 
/* The next set of functions all find an event to be mutated. Only   */
/* one of these gets used in a run; set by command-line arg -ett:..  */
/* Choices:                                                          */
/*   find_worst_event  finds an event with bad violation score by    */
/*                     using roulette-wheel selection                */
/*   find_random_event picks any one                                 */
/*   find_event_by_tn  uses tournament selection, size tt_mtn1       */
/*********************************************************************/ 

int find_worst_event(geno)
char *geno;
{
  int i;
  double penalty, p;

  /* Look at each event (=gene locus) and consider the constraints   */
  /* on it. Add up penalties, note violation score, accumulate total */
  /* violation score (CLASH and CLASH_COPY both count in this) for   */
  /* roulette selection of a gene to mutate.                         */

  penalty = 0.0;
  for(i = 0; i < geno_size; i++) {
    p = violation_score(i,SLOTVAL(geno[i]),geno);
    penalty += p;
    tt_scores[i] = p;
  } /* end of looping through genotype */
  
  /* Now we do roulette-wheel selection to choose a gene to mutate */

  /* Randomly scale down penalty */
  penalty = drandom()*penalty;
  i = 0;
  p = 0.0;
  /* Loop to find when accumulated violation scores exceed penalty */
  while(i < geno_size) {
    p += tt_scores[i];
    if(p >= penalty) /* >= is used here, just in case penalty = 0.0 */
      break;
    else i++;
  }
  return(i);
}

int find_random_event(geno)
char *geno;
{
  return((int)(drandom()*(double)geno_size));
}

int find_event_by_tn(geno)
char *geno;
{
  int event, j, e;
  double worst, penalty;
  /* Use tournament selection to find reasonably troublesome event */

  event = (int)(drandom()*(double)geno_size);
  worst = violation_score(event,SLOTVAL(geno[event]),geno);
  for(j=0;j<tt_mtn1-1;j++) {
    e = (int)(drandom()*(double)geno_size);
    penalty = violation_score(e,SLOTVAL(geno[e]),geno);
    if(penalty > worst) {
      worst = penalty;
      event = e;
    }
  }
  return(event);
}

/*********************************************************************/ 
/* The next set of functions all find a decent slot for the event.   */
/* One of these gets used in a run; set by command-line arg -ett:..  */
/* Choices:                                                          */
/*   find_free_slot    finds an unconstrained slot, else random      */
/*   find_random_slot  picks any one                                 */
/*   find_slot_by_tn   uses tournament selection, size tt_mtn2       */
/*********************************************************************/ 

int find_free_slot(geno,i)
char *geno;
int i;
{
  int *slotflags;
  int j, slot, otherslot, okslots;
  struct constraint *ctmp;

  /* What do we mutate its slot to? Preferably a slot that's */
  /* trouble-free, being not constrained at all. Otherwise   */
  /* something random.                                       */

  /* Set up an array of flags: 1 = bad slot, 0 = good slot   */
  slotflags = (int *)malloc(tt_maxslots*sizeof(int));
  for(slot=0;slot<tt_maxslots;slot++) slotflags[slot] = 0;
  slotflags[SLOTVAL(geno[i])] = 1; /* Not current one!   */

  /* Now loop through constraints on event i, and mark bad   */
  /* slots.                                                  */
  for(ctmp = tt_event_data[i];
      ctmp != (struct constraint *)NULL;
      ctmp = ctmp->next) {
    switch(ctmp->ctype) {
      case CLASH:
      case CLASH_COPY:
        slotflags[SLOTVAL(geno[ctmp->e2])] = 1;
        break;
      case BEFORE:
        otherslot = SLOTVAL(geno[ctmp->e2]);
        for(j=otherslot;j<tt_maxslots;j++) slotflags[j] = 1;
        break;
      case EXCLUDE:
        slotflags[ctmp->s] = 1;
        break;
      default:
        fprintf(stderr,"PGA: mystery constraint type when mutating\n");
        exit(1);
        break;
    }
  }

  /* Find the number of trouble-free slots. In the loop above     */
  /* some slots might be marked several times, so it's probably   */
  /* as cheap just to run through the array once more as it would */
  /* be to try to do the counting during the above loop.          */
  okslots = 0;
  for(slot=0;slot<tt_maxslots;slot++)
    if(slotflags[slot] == 0) okslots++;

  if(okslots > 0) {
    /* There are some slots where there seems to be no trouble at all */
    /* Pick one .. scale number down randomly .. */
    okslots = (int)(drandom()*(double)okslots);
    /* .. and go locate it: */
    for(slot=0;slot<tt_maxslots;slot++) {
      if(slotflags[slot] == 0) {
      	if(--okslots) break;
      }
    }
    /* Now slot is the new slot for event i */
  } else {
    /* No trouble-free slots here at all, pick a random one */
    slot = (int)(drandom()*(double)tt_maxslots);
  }
  free(slotflags);
  return(slot);
}

int find_random_slot(geno,i)
char *geno;
int i;
{
  return((int)(drandom()*(double)max_allele));
}

int find_slot_by_tn(geno,i)
char *geno;
int i;
{
  double penalty, bestp;
  int best, j, k;

  /* What do we mutate its slot to? Try some tournament selection */
  /* to find out.                                                 */
  
  best = (int)(drandom()*(double)max_allele);
  bestp = violation_score(i,best,geno);
  for(j=0;j<tt_mtn2-1;j++) {
    k = (int)(drandom()*(double)max_allele);
    penalty = violation_score(i,k,geno);
    if(penalty < bestp) {
      bestp = penalty;
      best = k;
    }
  }
  return(best);
}

/*******************************************************************/
/* This does the actual mutation of a timetable. The mutation rate */
/* is the probability of changing one event's slot; only one event */
/* gets moved per call, to avoid deadlock. So it's really a rate   */
/* of chromosomal mutation rather than bit mutation.               */
/*******************************************************************/

CHROMOSOME mutate_tt(original,rate)
CHROMOSOME original;
double rate;
{
  int i, slot;
  char *geno = original.gdp->genotype;
  CHROMOSOME newchromo;

  /* In timetabling, rate is taken to be the chance of making a      */
  /* single mutation of a whole chromosome; it makes less sense to   */
  /* randomly alter bits than to change a bad bit to a good one.     */
  if(drandom() > rate)
    return(original);
  
  /* Find event to move */
  i = find_bad_event(geno);  

  /* Find slot to move it to */
  slot = find_good_slot(geno,i);

  /* Is original wanted already? If so, set up new one else use original */
  /* It is not this routine's job to decide whether the resulting        */
  /* mutated chromosome is wanted at all; after all, might be very unfit */
  if(original.gdp->refcount > 0) {
     newchromo.gdp = (GENODATA *)malloc(sizeof(GENODATA));
     newchromo.gdp->refcount = 0;
     newchromo.gdp->genotype = (char *)malloc(geno_size+1);
     newchromo.gdp->genotype[geno_size]='\0';
     newchromo.gdp->id = chromo_id++;
     newchromo.gdp->parent1 = original.gdp->parent1;
     newchromo.gdp->parent2 = original.gdp->parent2;
     strncpy(newchromo.gdp->genotype,original.gdp->genotype,geno_size);
  } else
     newchromo = original;

  newchromo.gdp->genotype[i] = slot+'0';  	

  return(newchromo);
  
}

