/************************************************************************
 *                                                                      *
 *               PGA - Parallel Genetic Algorithm Testbed               *
 *                                                                      *
 *           "A Parallel Implementation of a genetic algorithm"         *
 *                                                                      *
 *                By Peter Ross (peter@ed.ac.uk),                       *
 *                   Geoffrey H. Ballinger (geoff@ed.ac.uk)             *
 *                                                                      *
 ************************************************************************/

/* 
   Copyright (C) 1993, 1994 Peter Ross and Geoffrey H. Ballinger.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
*/

#include <stdio.h>            /* Include the IO and maths header files. */
#include <math.h>
#include <curses.h>

#include "version.h"          /* title string and version number */
#include "pga.h"              /* defines genodata and chromosome structs */
#include "screen.h"           /* screen layout configuration */

void handle();                /* Prototypes; defined in this file. */
int report();
void setprompt();
void free_pops();
int compchromo();
void insert();
void dump_chromos();
double drandom();

/* Defined in init.c */

extern CHROMOSOME *init_pops_sorted();
extern CHROMOSOME *init_pops_unsorted();

/* Defined in eval.c */

extern double eval_max();
extern double eval_dj1();
extern double eval_dj2();
extern double eval_dj3();
extern double eval_dj4();
extern double eval_dj5();
extern double eval_bf6();
extern double eval_mcb();
extern double eval_knap();
extern int read_tt_datafile();
extern double eval_tt();
extern int read_weights_file();
extern double eval_rr();
extern void read_rrdata_file();
extern void long_decode();
extern void double_decode();

/* Defined in select.c */

extern CHROMOSOME rank_select();
extern CHROMOSOME fitprop_select();
extern CHROMOSOME tm_select();
extern CHROMOSOME tn_select();

/* Defined in reprod.c */

extern void one_reproduction();
extern void gen_reproduction();
extern void ss_gen_reproduction();
extern void ss_one_reproduction();

/* Derfined in cross.c */

extern CHROMOPAIR one_pt_cross();
extern CHROMOPAIR two_pt_cross();
extern CHROMOPAIR uniform_cross();
extern CHROMOPAIR no_cross();

/* Defined in mutate.c */

extern CHROMOSOME mutate_01();
extern CHROMOSOME mutate_multi();
extern CHROMOSOME mutate_tt();
extern double adaptive();
extern double nonadaptive();
/* Use one of these in timetable mutation...*/
extern int find_worst_event();
extern int find_random_event();
extern int find_event_by_tn();
/* ...and also one of these: */
extern int find_free_slot();
extern int find_random_slot();
extern int find_slot_by_tn();

/* Defined in help.c */

extern void pga_help();



/* Standard C lib stuff */

extern FILE *fopen();
extern int fscanf();
extern char *malloc();
extern char *calloc();
extern char *strdup();
extern long random();
#if SOLARIS
extern int sprintf();
#else
extern char *sprintf();
#endif SOLARIS

/* Globals used in various files, but declared here  */

int numpop = 5,            /* Number of populations */
    numchromo = 50,        /* Number of chromos in a population */
    geno_size = 32,        /* Length of a chromosome */
    gens_stage = 100,      /* How many times to cycle before asking user */
    repinterval = 10,      /* Interval between screen/file reports */
    isadaptive = FALSE,    /* Using adaptive mutation? */
    usegray = FALSE,       /* Treat bit patterns as Gray codes in decode */
    ss_task = FALSE,       /* A spatial selection run (ssgen or ssone)? */
    pause_first = FALSE,   /* Allow user to read complaints about tt file? */
    last_line = 24,        /* Size of screen; actually found at run time */
    rr_blocks,             /* RR number of low-level blocks (no default) */
    rr_blocksize = 8,      /* RR default block size */
    rr_gap = 7,            /* RR default gap */
    rr_mstar = 4,          /* RR default no of rewardable bits, low-level */
    migint = 10,           /* Interval between migrations */
    max_allele = 1,        /* In most problems, alleles in range 0..this */
    tournament_size = 10,  /* Default size in tournament selection */
    tt_maxevents = 0,      /* Default # of events in timetabling */
    tt_maxslots = 0,       /* Defualt # of slots in timetabling */
    tt_days = 1,           /* Default # of days in timetabling */
    tt_slots_per_day = 1,  /* Default # of slots per day in tiemtabling */
    tt_spread = 1,         /* Default definition of near-clash */
    tt_clash_type = 1,     /* Clashes don't count per-student (and gripe) */
    tt_mtn1 = 3,           /* Event tournament size in timetable mutation */
    tt_mtn2 = 3,           /* Slot tournament size in timetable mutation */
    chromo_id = 1;         /* Chromosome IDs start from this value */
int randseed;              /* The random seed */
long *sticks;              /* Used in knapsack prob: array of sizes */
long target;               /* Used in knapsack prob: target */
FILE *output = (FILE *)NULL;       /* Output file if any */
FILE *chromofile = (FILE *)NULL;   /* Chromo file, if any output file */
int interactive = TRUE;            /* Interactive? */
int finaldump = FALSE;             /* Noninteractive: do final chromodump? */
int chromoutput = FALSE;           /* Any chromosome output? */
int twins = FALSE;         /* Crossover produces pairs, or just one child? */
char *datafilename = (char *)NULL;
char outfilename[256], chromofilename[256];
char prompt[80];
double *tt_scores;         /* Used in timetabling, smart mutation */
char *geno_size_string;    /* String naming size of a chromo (hack) */
char *rr_blockarray;       /* Used in RR for higher-level block scoring */
double mute_rate = 0.02,   /* Default mutation rate */
       cross_rate = 0.6,   /* Default crossover rate if applicable */
       bias = 1.5,         /* Default bias for rank selection */
  tt_penalty_clash = 10.0, /* TT: penalty for a clash */
  tt_penalty_order = 3.0,  /* TT: penalty for an order violation */
  tt_penalty_exclude = 5.0,/* TT: penalty for an exclusion violation */
  tt_penalty_consec = 1.0, /* TT: penalty for a near-clash */
       rr_ustar = 1.0,     /* RR default reward for first block */
       rr_u = 0.3,         /* RR default reward for later blocks */
       rr_v = 0.02;        /* RR default bonus/penalty for low-level bit */
char *eval_name,           /* String giving name of problem */
     *maxfitness,          /* String giving max fitness (if known) */
     *cross_name,          /* String naming crossover type */
     *select_name,         /* String naming selection type */
     *repro_name;          /* String naming reproduction type */
double *total_abs_fitness; /* Used in roulette selection: size of wheel */
int evals;                 /* How often eval has been called */
int ss_xmax,ss_ymax,ss_n;  /* Size of grid and length of walk in ssN */
double user_threshold;     /* Non-interactive? Stop when one/all reach this */
int n_for_stop;            /* How many must reach user_threshold to stop */

/* Timetabling structures */

/* This will be an array where the chains get rooted: */

struct constraint **tt_event_data = (struct constraint **)NULL;

/* And this is where the preset events get noted: */

struct constraint *tt_presets = (struct constraint *)NULL;

/* This struct matches keywords to symbols, and says how many numbers */
/* ought to be picked up after the keyword is found. A value of ALL   */
/* in the args_wanted field means gather up as many numbers as can be */
/* found on the input line, rather than a fixed number.               */

#define CHUNK 10  /* The array where numbers get stashed first will    */
                  /* increase by this size whenever necessary. MUST BE */
                  /* AT LEAST 2.                                       */
#define ALL -1    /* MUST BE NEGATIVE: means "get all numbers on line" */

struct tt_params tt_paramlist[] = {
  {"events",       DATA_EVENTS,         1  },
  {"slots",        DATA_SLOTS,          1  },
  {"days",         DATA_DAY,            1  },
  {"near-clash",   DATA_SPREAD,         1  },
  {"clash-type",   DATA_CLASHTYPE,      1  },
  {"p-clash",      PENALTY_CLASH,       1  },
  {"p-order",      PENALTY_ORDER,       1  },
  {"p-exclude",    PENALTY_EXCLUDE,     1  },
  {"p-near",       PENALTY_NEARCLASH,   1  },
  {"separate",     CONSTRAIN_CLASH,     ALL},
  {"order",        CONSTRAIN_BEFORE,    ALL},
  {"exclude",      CONSTRAIN_EXCLUDE,   ALL},
  {"preset",       CONSTRAIN_PRESET,    2  },
  {NULL, END_LIST, 0  }
};


/* Default bindings for function pointers */

CHROMOSOME *(*initialise)() = init_pops_sorted;
CHROMOSOME (*selection)() = rank_select;
CHROMOPAIR (*crossover)() = two_pt_cross;
CHROMOSOME (*mutate)() = mutate_01;
void (*reproduction)() = one_reproduction;
void (*decode)() = long_decode;
double (*mrate)() = nonadaptive;
double (*evaluate)() = eval_max;
int (*find_bad_event)() = find_worst_event;
int (*find_good_slot)() = find_free_slot;


/************************************************************************/
/* The main program.                                                    */
/************************************************************************/

main(argc,argv)
int argc;
char *argv[];
{
  int command,chromo,pop,gen,mignext,action,gen_total,fileoutput,i;
  double average;
  char c;
  int mapped, saved, converged, satisfied;
  CHROMOSOME *pops,migchromo;

  randseed = (int) time(0);

  mignext=0;
  n_for_stop = 0;
  satisfied = FALSE;
  geno_size_string = (char *)malloc(20);
  sprintf(geno_size_string,"%d",geno_size);
  maxfitness = geno_size_string;
  eval_name = "max";
  cross_name = "two-point";
  select_name = "rank";
  repro_name = "one";  

  
  for (command=1; command<argc; command+=1) {
    handle(argv[command]);                    /* Handle any arguments. */
  }

  srandom(randseed);

  if(interactive || n_for_stop < 0) /* If interactive, stop test is that ALL */
    n_for_stop = numpop;            /* have converged; user may say sooner.. */

  if(geno_size > CRUCIAL_GENO_SIZE) {
    decode = double_decode;
    if(usegray) {
      fprintf(stderr, "Chromosome length looks a bit big for Gray code\n");
      fprintf(stderr, "option - admittedly just a crude check.\n");
      exit(1);
    }
  }
  if(evaluate == eval_knap) {
    sticks = (long *)calloc(geno_size,sizeof(long));
    if(datafilename == (char *)NULL)
      datafilename = WEIGHTSFILE;
    if(!read_weights_file()) {
      fprintf(stderr,"No weights file accessible. It should hold\n");
      fprintf(stderr,"the target value (an integer) and up to\n");
      fprintf(stderr,"%d further integers. The aim is to find a\n", geno_size);
      fprintf(stderr,"subset with sum as close to the target as possible.\n");
      exit(1);
    }
  } else if(evaluate == eval_rr) {
      int n;
      double max;
      if(datafilename == (char *)NULL)
        datafilename = RRDATAFILE;
      read_rrdata_file();
      rr_blockarray = malloc(rr_blocks);
      maxfitness = (char *)malloc(20);
      n = rr_blocks;
      max = 0.0;
      while(n>0) {
        max += rr_ustar + (n-1)*rr_u;
        n /= 2;
      }
      sprintf(maxfitness,"%.6f", max);
  } else if(evaluate == eval_tt) {
      if(datafilename == (char *)NULL)
        datafilename = TTDATAFILE; 
      if(!read_tt_datafile()) {
        fprintf(stderr,"See the documentation for details of the data file\n");
        exit(1);
      }
  }

  if((reproduction == ss_gen_reproduction)
     || (reproduction == ss_one_reproduction)) {
        ss_task = TRUE;
        ss_xmax = (int) sqrt((double)numchromo);
        ss_ymax = numchromo/ss_xmax;
        numchromo = ss_xmax*ss_ymax;
        migint = 0;
  }

  if(mutate == mutate_tt) {
    tt_scores = (double *)malloc(geno_size*sizeof(double));
  }

  if(pause_first) {
    fprintf(stderr,"**** Press RETURN to continue..");
    while(getchar() != '\n');
  }

  fileoutput = (output == (FILE *)NULL)? FALSE : TRUE;


  if(interactive) {
    stdscr = initscr();
    last_line = tgetnum("li");
    if(last_line < 0) last_line = 24;
  
    mvprintw(POP_ROW,POP_COL,      "          Populations: %-6d",numpop);
    if((reproduction == ss_gen_reproduction)
       || (reproduction == ss_one_reproduction))
      mvprintw(CPP_ROW,CPP_COL,      "  Chromosomes per pop: %-4d (%dx%d)",
                                     numchromo,ss_xmax,ss_ymax);
    else
      mvprintw(CPP_ROW,CPP_COL,      "  Chromosomes per pop: %-4d",numchromo);
    mvprintw(EVLIM_ROW,EVLIM_COL,  "Generations per stage: %-6d",gens_stage);
    mvprintw(REPINT_ROW,REPINT_COL,"   Reporting interval: %-6d",repinterval);
    mvprintw(MIG_ROW,MIG_COL,      "   Migration interval: %-6d",migint);
    mvprintw(CTYPE_ROW,CTYPE_COL,  "   Crossover type: %s, %s", cross_name,
                                            (twins)? "twins" : "one child");
    if(!strcmp(repro_name,"one")
       || !strncmp(repro_name,"ssone",5))
      mvprintw(CROSS_ROW,CROSS_COL,"   Crossover rate: n/a");
    else
      mvprintw(CROSS_ROW,CROSS_COL,"   Crossover rate: %3.2f",cross_rate);
    if(!strcmp(select_name,"rank"))
      mvprintw(BIAS_ROW,BIAS_COL,  "   Selection bias: %3.2f",bias);
    else
      mvprintw(BIAS_ROW,BIAS_COL,  "   Selection bias: n/a");
    if(isadaptive)
      mvprintw(MUT_ROW,MUT_COL,    "    Mutation rate: adaptive %.4f",
                                                                mute_rate);
    else
      mvprintw(MUT_ROW,MUT_COL,    "    Mutation rate: %.6f",mute_rate);
    mvprintw(CRL_ROW,CRL_COL,      "Chromosome length: %d",geno_size);
    if(fileoutput)
      mvprintw(FNAME_ROW,FNAME_COL,"    Log file name: %s",outfilename);
    mvprintw(EVAL_ROW,EVAL_COL,    "Eval function: %s",eval_name);
    mvprintw(SELECT_ROW,SELECT_COL,"    Selection: %s", select_name);
    mvprintw(REPRO_ROW,REPRO_COL,  " Reproduction: %s",repro_name);
    mvprintw(POPTABLE_ROW,POPTABLE_COL,
                 "Pop.......Average..........Best.(max = %s)",maxfitness);
    mvprintw(GEN_ROW,GEN_COL,"Generation: ");
    clrtoeol();
    mvprintw(EVCOUNT_ROW,EVCOUNT_COL,"Evaluations so far: ");
    clrtoeol();
    refresh();
    cbreak();
    noecho();
  }

  total_abs_fitness = (double *)malloc(numpop*sizeof(double));
  pops = initialise(numpop,numchromo);
  gen_total = 1;
  evals=0;
  action = CONTINUE;

  if(fileoutput) {
    fprintf(output,   "\n\n===================================\n");
    fprintf(output,   "               eval = %s\n",eval_name);
    fprintf(output,   "        max-fitness = %s\n",maxfitness);
    fprintf(output,   "       reproduction = %s\n",repro_name);
    fprintf(output,   "             select = %s\n",select_name);
    if(!strcmp(select_name,"fitprop"))
      fprintf(output, "     selection-bias = n/a\n");
    else
      fprintf(output, "     selection-bias = %.6f\n",bias);
    fprintf(output,   "  chromosome-length = %d\n",geno_size);
    fprintf(output,   "               pops = %d\n",numpop);
    if((reproduction == ss_gen_reproduction)
       || (reproduction == ss_one_reproduction))
      fprintf(output,   "chromosomes-per-pop = %d : %d x %d\n",
                        numchromo,ss_xmax,ss_ymax);
    else
      fprintf(output,   "chromosomes-per-pop = %d\n",numchromo);
    fprintf(output,   " migration-interval = %d\n",migint);
    fprintf(output,   "     crossover-type = %s\n",cross_name);
    if(!strcmp(repro_name,"one")
      || !strncmp(repro_name,"ssone",5))
      fprintf(output, "     crossover-rate = n/a\n");
    else
      fprintf(output, "     crossover-rate = %.6f\n",cross_rate);
    if(isadaptive)
      fprintf(output, "      mutation-rate = adaptive\n");
    else
      fprintf(output, "      mutation-rate = %.6f\n", mute_rate);
    fprintf(output,   " reporting-interval = %d\n", repinterval);
    if(twins)
      fprintf(output, "              twins = yes\n");
    else
      fprintf(output, "              twins = no\n");
    fprintf(output,   " random-number-seed = %d\n", randseed);
    fprintf(output,   "\n");
  }
  converged = report(0,fileoutput,pops,user_threshold);
  mapped = saved = FALSE;
  i = 0; /* used to gather up any digits typed in */

  while(action != QUIT) {
      if(interactive)
      do {
              setprompt(fileoutput,ss_task,mapped,saved,satisfied);
              mvprintw(0,1,prompt);
              clrtoeol();
              refresh();
              switch(c=getchar()) {
                    case 'a': case 'A':
                            printw("starting again..");
			    refresh();
                            move(GEN_ROW,GEN_COL+12);
                            clrtoeol();
                            move(EVCOUNT_ROW,EVCOUNT_COL+20);
                            clrtoeol();
                            refresh();
                            free_pops(pops,numpop*numchromo);
                            gen_total = 1;
                            pops = initialise(numpop,numchromo);
                            evals = 0;
                            if(fileoutput) {
                              fprintf(output,
                                      "\n\n------ re-starting ------\n\n");
			    }
                            converged=report(0,fileoutput,pops,user_threshold);
                            satisfied = (converged >= n_for_stop)? TRUE:FALSE;
                            mapped = saved = FALSE;
                            action = ASK_AGAIN;
                            i = 0; /* reset number input */
                            break;
                    case 'c': case 'C':
                            if(converged == numpop)
                              action = ASK_AGAIN;
                            else {
                              printw("continuing.."); refresh();
                              action = CONTINUE;
                              mapped = saved = FALSE;
                              if(i>0) {
                                 gens_stage = i;
			         mvprintw(EVLIM_ROW,EVLIM_COL,
		  			  "Generations per stage: %-6d",
                                          gens_stage);
                                 i=0;
			      }
			    }
                            break;
                    case 'q': case 'Q':
                            action = QUIT;
                            break;
		    case 's': case 'S':
                            if(fileoutput && !saved) {
		               printw("saving chromosomes.."); refresh();
                               dump_chromos(pops,gen_total-1);
			       saved = TRUE;
			    } else if(!saved) {
                               strcpy(outfilename,"unnamed");
			       printw("saving chromosomes in unnamed.chr.."); 
                               refresh();
                               dump_chromos(pops,gen_total-1);
			       saved = TRUE;
			    }
                            action = ASK_AGAIN;;
                            i = 0; /* reset number input */
                            break;
		    case 'm': case 'M':
			    printw("writing grid map.."); refresh();
                            if(fileoutput && ss_task && !mapped) {
 			      ss_map(outfilename,
                                     (gen_total-1),pops,numpop,numchromo);
			      mapped = TRUE;
			    }
                            action = ASK_AGAIN;
                            i = 0; /* reset number input */
                            break;
		    case '0':
		    case '1':
		    case '2':
		    case '3':
		    case '4':
		    case '5':
		    case '6':
		    case '7':
		    case '8':
		    case '9':
			    i = 10*i + c - '0';
                            action = ASK_AGAIN; 
                            break;
                    default:
                            action = ASK_AGAIN; 
                            i = 0; /* reset number input */
                            break;
              }
      } while(action == ASK_AGAIN);
      if(action == CONTINUE) {
        for(gen=1; gen <= gens_stage && !satisfied; gen++, gen_total++) {
          for(pop=0; pop<numpop; pop+=1) {            /* For each generation */
            reproduction(pops,pop*numchromo,numchromo);        /* Reproduce. */
          }
          if (numpop>1 && migint > 0 && gen%migint==0) {       /* Migrate if */
            migchromo=selection(pops,mignext*numchromo,numchromo);
            for(pop=0; pop<numpop; pop+=1) {                   /* necessary. */
              if(pop != mignext)               /* Don't insert to its source */
                insert(pops,migchromo,pop*numchromo,numchromo);
            }
            mignext=(mignext+1)%numpop;
          }
          if (gen_total%repinterval==0) {                      /* Report */
            converged = report(gen_total,fileoutput,pops,user_threshold);
  	  }
          if(converged >= n_for_stop)
            satisfied = TRUE;
        }
        if((gen_total-1)%repinterval != 0) { /* Last report a while ago? ... */
          converged =
            report(gen_total-1,fileoutput,pops,user_threshold); /* ..report. */
          if(converged >= n_for_stop)
            satisfied = TRUE;
	}
        if(!interactive) {
          if(fileoutput && finaldump)
            dump_chromos(pops,gen_total-1);            
	  action = QUIT;
	}
      }      
  }

  /* All done, tidy up */
  if(interactive) {
    echo();
    nocbreak();
    move(last_line-1,0);
    refresh();
    endwin();
  }

  if(fileoutput)
     fclose(output);  
  if(chromoutput)
     fclose(chromofile);

  free_pops(pops,numpop*numchromo);
  exit(0);
}

/************************************************************************/
/* This sets the prompt for next interaction                            */
/************************************************************************/

int stringcat(s1,s2,n)
char *s1, *s2;
int n;
{
  int i,j;
  for(i=n,j=0; s2[j] != '\0'; i++, j++) {
    s1[i] = s2[j];
  }
  s1[i] = '\0';
  return(i);
}

void setprompt(filing,sstask,mapped,saved,satisfied)
int filing,sstask,mapped,saved,satisfied;
{
  int i;
  for(i=0;i<80;i++) prompt[i]='\0';

  i=stringcat(prompt,"(A)gain, (Q)uit", 0);
  if(filing && !saved)  i=stringcat(prompt, ", (S)ave", i);
  if(sstask && !mapped) i=stringcat(prompt, ", (M)ap", i);
  if(!satisfied)        i=stringcat(prompt, ", (C)ontinue",i);
  stringcat(prompt, ": ", i);
}

/************************************************************************/
/* This updates the screen and, if necessary, the output file           */
/* If interactive, returns the number that have converged.              */
/* If NOT interactive, returns the number that have passed the supplied */
/* threshold value.                                                     */
/************************************************************************/

int report(gen_total,fileoutput,pops,threshold)
int gen_total, fileoutput;
CHROMOSOME *pops;
double threshold;
{
  int pop, chromo;
  int n_at_threshold = 0;
  char conv_char;
  double average, best, tmp;

  if(interactive) {
    mvprintw(GEN_ROW,GEN_COL+12,"%d",gen_total);         /* necessary */
    mvprintw(EVCOUNT_ROW,EVCOUNT_COL+20,"%d", evals);
  }
  if(fileoutput) {
    fprintf(output, "\nGenerations = %d  Evaluations-so-far = %d\n",
                    gen_total,evals);
  }

  for(pop=0; pop<numpop; pop+=1) {               
    average=0.0;
    best = -1000000.0;
    for(chromo=0; chromo<numchromo; chromo+=1) {
        /* Note that best is usually pops[pop*numchromo] since */
        /* the population is sorted - but not sorted with -rssN */
        tmp = pops[pop*numchromo+chromo].fitness;
        average+=tmp;
        if(best < tmp) best = tmp;
    }
    average/=(double)numchromo;
    if(fabs(average-best) < 1e-9) {
      conv_char = '=';
      n_at_threshold++;
    } else {
      conv_char = ' ';
      if(!interactive && best >= threshold)
        n_at_threshold++;
    }
    if(interactive)
      mvprintw(POPTABLE_ROW+1+pop,POPTABLE_COL,
               "%-4d%c    %12.7f     %12.7f",
               pop, conv_char,
                          average,
                                     best);
    if(fileoutput) {
      fprintf(output, "  %-4d    %12.7f     %12.7f\n",
                         pop,    average,   best);
    }
  }
  if(interactive)
    refresh();
  return(n_at_threshold);
}


/************************************************************************/
/* This frees whole population                                          */
/************************************************************************/

void free_pops(whole_pop,num)
CHROMOSOME *whole_pop;
{
  int i;

  for(i=0; i<num; i++) {
    /* Avoid freeing anything twice; matters for some operating systems */
    if(whole_pop[i].gdp->refcount == 1) {
      free(whole_pop[i].gdp->genotype);
      free(whole_pop[i].gdp);
    } else
      whole_pop[i].gdp->refcount--;
  }
  free(whole_pop);
}

/************************************************************************/
/* This resets the defaults according to the argument 'arg'.            */
/************************************************************************/

void handle(arg)
char *arg;
{
  int newint;
  double newdouble;
  char tt_eventchoice[16], tt_slotchoice[16];

  if (*arg=='-') {               /* Either a switch ...        */
    switch (*(arg+1)) {
    case 'P' :
      if(sscanf(arg,"-P%d",&newint) == 1) {
        if (newint<1) {
          fprintf(stderr,"PGA: You must have at least 1 population.\n");
          exit(1);
        }
        else {
          numpop=newint;
        }
      }
      break;
    case 'p' :
      if(sscanf(arg,"-p%d",&newint) == 1) {
        if (newint<2) {
          fprintf(stderr,
              "PGA: You must have at least 2 chromosomes per population.\n");
          exit(1);
        }
        else {
          numchromo=newint;
        }
      }
      break;
    case 'n' :
      if(sscanf(arg,"-n%d",&newint) == 1) {
        if (newint<12) {
          fprintf(stderr,"PGA: Chromosome length must be at least 12.\n");
          exit(1);
        }
        else {
          geno_size=newint;
          sprintf(geno_size_string,"%d",geno_size);
        }
      }
      break;
    case 'N':
      interactive = FALSE;
      if(strlen(arg) < 4) {
        fprintf(stderr,
         "PGA: non-interactive: must specify -NOthreshold or -NAthreshold\n");
        exit(1);
      }
      switch(*(arg+2)) {
        case 'o':
          finaldump = TRUE;
        case 'O':
          n_for_stop = 1;
          break;
        case 'a':
          finaldump = TRUE;
        case 'A':
          n_for_stop = -1; /* because numpop may not yet be set! */
          break;
        default:
          fprintf(stderr, 
                  "PGA: non-interactive: must say O (one) or A (all)\n");
  	  exit(1);
      }
      if(sscanf(arg+3,"%lf",&newdouble) == 1) {
        user_threshold = newdouble;
      } else {
        fprintf(stderr,"PGA: non-interactive: must set stop threshold\n");
        exit(1);
      }
      break;
    case 'l' :
      if(sscanf(arg,"-l%d",&newint) == 1) {
        if (newint<1) {
          fprintf(stderr,
                  "PGA: Less than one generation per stage isn't sensible!\n");
          exit(1);
        }
        else {
          gens_stage=newint;
        }
      }
      break;
    case 'i' :
      if(sscanf(arg,"-i%d",&newint) == 1) {
        if (newint<1) {
          fprintf(stderr,
                "PGA: A report every %d generations isn't sensible!\n",newint);
          exit(1);
        }
        else {
          repinterval=newint;
        }
      }
      break;
    case 'M' :
      if((reproduction == ss_gen_reproduction)
         || (reproduction == ss_one_reproduction)) {
        fprintf(stderr, 
                "PGA: cannot use -M or -s with -rssoneN or -rssgenN\n");
        exit(1);
      }
      if(sscanf(arg,"-M%d",&newint) == 1) {
        if (newint<0) {
          fprintf(stderr,
                  "PGA: A migration interval of %d makes no sense.\n",newint);
          exit(1);
        }
        else {
          migint=newint;
        }
      }
      break;
    case 'S' :
      if(sscanf(arg,"-S%d",&newint) == 1) {
        if(newint<0) {
          fprintf(stderr,"PGA: random seed must be non-negative, set to 1.\n");
          newint = 1;
        }
        randseed = newint;
      }
      break;
    case 'a' :
      isadaptive=!isadaptive;
      if (isadaptive) mrate=adaptive;
      else mrate=nonadaptive;
      break;
    case 'g' :
      usegray = TRUE;
      break;
    case 't' :
      twins = !twins;
      break;
    case 'F' :
      datafilename=strdup(arg+2);
      if(strlen(datafilename) == 0) {
        fprintf(stderr,"PGA: use -Fdatafile, no space between F and name\n");
        exit(1);
      }
      break;
    case 'm' :
      if(sscanf(arg,"-m%lf",&newdouble) == 1) {
        if (newdouble<0 || newdouble>1) {
          fprintf(stderr,"PGA: The mutation rate must be between 0 and 1.\n");
          exit(1);
        }
        else {
          mute_rate=newdouble;
        }
      }
      break;
    case 'c' :
      if(sscanf(arg,"-c%lf",&newdouble) == 1) {
        if (newdouble<0 || newdouble>1) {
          fprintf(stderr,"PGA: The crossover rate must be between 0 and 1.\n");
          exit(1);
        }
        else {
          cross_rate=newdouble;
        }
      }
      break;
    case 'b' :
      if(sscanf(arg,"-b%lf",&newdouble) == 1) {
        if (newdouble<1.0 || newdouble>2.0) {
          fprintf(stderr,"PGA: The bias must be strictly between 1 and 2.\n");
          exit(1);
        }
        else {
          bias=newdouble;
        }
      }
      break;
    case 's' :
      if((reproduction == ss_gen_reproduction)
         || (reproduction == ss_one_reproduction)) {
        fprintf(stderr, "PGA: cannot use -s or -M with -rssN\n");
        exit(1);
      }
      if (!strcmp(arg,"-srank")) {
        selection = rank_select;
        select_name = "rank";
      }
      else if (!strcmp(arg,"-sfitprop")) {
        selection = fitprop_select;
        select_name = "fitprop";
      }
      else if (!strncmp(arg,"-stm",4)) {
        selection = tm_select;
        select_name = strdup(arg+2);
        tournament_size = atoi(arg+4);
        if (tournament_size < 1) {
          fprintf(stderr,"PGA: tm selection requires arg > 1; set to 10\n");
          tournament_size = 10;
        }
      }
      else if (!strncmp(arg,"-stn",4)) {
        selection = tn_select;
        select_name = strdup(arg+2);
        tournament_size = atoi(arg+4);
        if (tournament_size < 1) {
          fprintf(stderr,"PGA: tn selection requires arg > 1; set to 10\n");
          tournament_size = 10;
        }
      }
      else {
        fprintf(stderr,"PGA: I don't know about %s selection.\n",arg+2);
        exit(1);
      }
      break;
    case 'r' :
      if (!strcmp(arg,"-rone")) {
        reproduction = one_reproduction;
        repro_name = "one";
      }
      else if (!strcmp(arg,"-rgen")) {
        reproduction = gen_reproduction;
        repro_name = "gen";
      }
      else if (!strncmp(arg,"-rssgen",7)) {
        reproduction = ss_gen_reproduction;
        repro_name = strdup(arg+2);
        initialise = init_pops_unsorted;
        select_name = "*spatial*";
        ss_n = atoi(arg+7);
        if(ss_n < 1) {
          fprintf(stderr, "PGA: ss random walk must be of length > 0, set to 5\n");
          ss_n = 5;
	}
      }
      else if (!strncmp(arg,"-rssone",7)) {
        reproduction = ss_one_reproduction;
        repro_name = strdup(arg+2);
        initialise = init_pops_unsorted;
        select_name = "*spatial*";
        ss_n = atoi(arg+7);
        if(ss_n < 1) {
          fprintf(stderr, "PGA: ss random walk must be of length > 0, set to 5\n");
          ss_n = 5;
	}
      }
      else {
        fprintf(stderr,"PGA: I don't know about %s reproduction.\n",arg+2);
        exit(1);
      }
      break;
    case 'C':
      if (!strcmp(arg,"-Cone")) {
        cross_name = "one-point";
        crossover = one_pt_cross;
      } else if (!strcmp(arg,"-Ctwo")) {
        cross_name = "two-point";
        crossover = two_pt_cross;
      } else if (!strcmp(arg,"-Cuniform")) {
        cross_name = "uniform";
        crossover = uniform_cross;
      } else if (!strcmp(arg,"-Cnone")) {
        cross_name = "none";
        crossover = no_cross;
      }
      break;
    case 'e' :
      eval_name = strdup(arg+2);
      if (!strcmp(arg,"-emax")) {
        evaluate = eval_max;
        sprintf(geno_size_string,"%d",geno_size);
        maxfitness = geno_size_string;
      }
      else if (!strcmp(arg,"-edj1")) {
        evaluate = eval_dj1;
        maxfitness = "100";
      }
      else if (!strcmp(arg,"-edj2")) {
        evaluate = eval_dj2;
        maxfitness = "1000";
      }
      else if (!strcmp(arg,"-edj3")) {
        evaluate = eval_dj3;
        maxfitness = "55";
      }
      else if (!strcmp(arg,"-edj5")) {
        evaluate = eval_dj5;
        maxfitness = "approx 499.001997";
      }
      else if (!strcmp(arg,"-ebf6")) {
        evaluate = eval_bf6;
        maxfitness = "approx 0.994007";
      }
      else if (!strcmp(arg,"-eknap")) {
        evaluate = eval_knap;
        maxfitness = "1.0";
      }
      else if (!strncmp(arg,"-ett",4)) {
        evaluate = eval_tt;
        mutate = mutate_multi;
        if(strlen(arg) > 4 && *(arg+4) == ':') {
          mutate = mutate_tt;
          if(sscanf(arg+5,"%[^+]+%[^+]",tt_eventchoice,tt_slotchoice) != 2) {
            fprintf(stderr,"PGA: mangled tt argument %s\n", arg);
            exit(1);
          } else {
            switch(tt_eventchoice[0]) {
	    case 'r':  /* random */
              find_bad_event = find_random_event;
              break;
	    case 'w':
              find_bad_event = find_worst_event;
              break;
            case 't':
              find_bad_event = find_event_by_tn;
              tt_mtn1 = atoi(tt_eventchoice+1);
              if(tt_mtn1 < 1) {
                fprintf(stderr,"PGA: -ett:tN+.. has silly N\n");
                exit(1);
	      }
              break;
            default:
	      fprintf(stderr,"PGA: -ett:.. unrecognised event chooser\n");
              exit(1);
	    }
            switch(tt_slotchoice[0]) {
	    case 'r':  /* random */
              find_good_slot = find_random_slot;
              break;
	    case 'f':
              find_good_slot = find_free_slot;
              break;
            case 't':
              find_good_slot = find_slot_by_tn;
              tt_mtn2 = atoi(tt_slotchoice+1);
              if(tt_mtn2 < 1) {
                fprintf(stderr,"PGA: -ett:..+tN has silly N\n");
                exit(1);
	      }
              break;
            default:
	      fprintf(stderr,"PGA: -ett:.. unrecognised slot chooser\n");
              exit(1);
	    }
	  }
	}
        maxfitness = "1.0";
      }
      else if (!strcmp(arg,"-err")) {
        evaluate = eval_rr;
        /* NB, maxfitness set after arg processing, once params known */
      }
      else if (!strncmp(arg,"-emcb",5)) {
        evaluate = eval_mcb;
        maxfitness = geno_size_string;
        mutate = mutate_multi;
        max_allele = atoi(arg+5);
        if(max_allele < 1 || max_allele > 9) {
          fprintf(stderr,"PGA: mcbK: K should be in [1..9], defaulted to 2\n");
          max_allele = 2;
	}
      }
      else {
        fprintf(stderr,"PGA: unknown evaluation function: %s\n",
                        arg+2);
        exit(1);
      }
      break;
    default :
      if (*(arg+1)!='h') fprintf(stderr,"Unknown argument %s.\n\n",arg);
      pga_help();
      exit(0);
    }
  }
  else {                         /* ... or an output filename. */
    output=fopen(arg,"a");
    if (output==NULL) {
      fprintf(stderr,"PGA: can't open %s for writing.\n",arg);
      exit(1);
    }
    strcpy(outfilename,arg);
  }
}

/************************************************************************/
/* Dumps chromosomes to a file. Filename is outfilename.chr, eg foo.chr */
/************************************************************************/

void dump_chromos(pops,g)
CHROMOSOME *pops;
int g;
{
  int i,j;

  if(!chromoutput) {
     sprintf(chromofilename,"%s.chr",outfilename);
     chromofile = fopen(chromofilename,"a");
     chromoutput = (chromofile != (FILE *)NULL);
  }

  if(chromoutput) {
    for(i=0;i<numpop;i++) {
      for(j=0;j<numchromo;j++) {
        fprintf(chromofile,
                "%-2d %-4d %-5d %12.7f  %s  %5d %5d %5d\n",
                 i,   j,   g,    pops[i*numchromo+j].fitness,
                                 pops[i*numchromo+j].gdp->genotype,
                                 pops[i*numchromo+j].gdp->id,
                                 pops[i*numchromo+j].gdp->parent1,
                                 pops[i*numchromo+j].gdp->parent2);
      }
      fprintf(chromofile,"\n\n");
    }
  }
}    


/************************************************************************/
/* This function inserts 'chromo' into the correct place (sorted by     */
/* fitness) within the subpopulation defined by 'base' and 'range' of   */
/* the populations array 'pops'.                                        */
/************************************************************************/

void insert(pops,chromo,base,range)
CHROMOSOME *pops,chromo;
int base,range;
{
  int current, pop;

  pop = base/numchromo;         /* which pop we're hitting */

                                /* Can it be inserted at all? */
  if (chromo.fitness>pops[base+range-1].fitness) {
    chromo.gdp->refcount++;
    if(pops[base+range-1].gdp->refcount == 1) {
      total_abs_fitness[pop] -= fabs(pops[base+range-1].fitness);
      free(pops[base+range-1].gdp->genotype);
      free(pops[base+range-1].gdp);
    }
    current=base+range-2;       /* Start at the second last place. */
    do {                        /* If 'chromo' is fitter ... */
      if (chromo.fitness>pops[current].fitness) {
        pops[current+1]=pops[current]; /* ... then push it down one. */
      }
      else {
        total_abs_fitness[pop] += fabs(chromo.fitness) 
                                   - fabs(pops[current+1].fitness);        
        pops[current+1]=chromo;        /* ... else insert 'chromo' below */
        current=base;                  /* and finish.                    */
      }
      current-=1;               /* Work up the array until ... */
    } while (current>=base);    /* we get past the 'base'.     */
    if (chromo.fitness>pops[base].fitness) {
      total_abs_fitness[pop] += fabs(chromo.fitness) 
                                 - fabs(pops[base].fitness);
      pops[base]=chromo;        /* If chromo is the best put it in. */
    }
  } else {     /* It cannot be inserted, it is terrible */
      if(chromo.gdp->refcount==0) {
        free(chromo.gdp->genotype);
        free(chromo.gdp);
      }
  }
}


/************************************************************************/
/* Returns >0 if chromo2 is fitter than chromo1, <0 if chromo 1 is      */
/* fitter than chromo2, and 0 if they are equally fit. This is passed   */
/* to qsort as its comparison function.                                */
/************************************************************************/

int compchromo(chromo1,chromo2)
CHROMOSOME *chromo1,*chromo2;
{
  if((chromo2->fitness) > (chromo1->fitness))
     return(1);
  else if((chromo2->fitness) == (chromo1->fitness))
     return(0);
  else
     return(-1);
}

/***********************************************************************/
/* Returns a random double in range 0.0 to 1.0 (never 1.0 exactly)     */
/***********************************************************************/
double drandom()
{
  return((double)random()/2147483647.0);
}

/* end of pga.c */
