%%% File:   clique.pl
%%% Author: Peter Ross
%%% Date:   31 May 1994

%%% This simple interactive program builds ever-larger cliques from edge
%%% data. The key to it is as follows: suppose there IS a clique of
%%% size 4, namely (A,B,C,D). Then (A,C,D) and (B,C,D) will be cliques
%%% of size 3 and there will be an edge between A and B. So to find
%%% ALL cliques of size 4, we can start from ALL cliques of size 3.
%%% And so on.
%%%
%%% Some test data, from a REAL timetabling problem, is attached.
%%% It contains two cliques of size 10 and none larger.

%%% clique/2 facts get asserted by this code, so declare it dynamic

:- dynamic clique/2.

%%% Top level is find_maximal_cliques(-Integer), which returns the
%%% size of the largest clique(s), and prints it/them out. If you
%%% backtrack into this, you will get a list of the next smaller
%%% cliques and so on.

find_maximal_cliques(Max) :-
  process_cliques(2,Max),
  setof(Clique,clique(Max,Clique),L),
  write('Cliques of size '), write(Max), write(':'), nl,
  write_list(L).

write_list([]).
write_list([H|T]) :-
  write('   '), write(H), nl,
  write_list(T).

%%% Grow ever-larger cliques one step at a time:

process_cliques(N,Max) :-
  extend_cliques(N,N1),
  process_cliques(N1,Max).
process_cliques(Max,Max).

%%% To build a bigger clique:
%%%   - we must have those of size N
%%%   - and none of size N+1
%%% in which case we can build all cliques of size N+1.

extend_cliques(N,N1) :-
  N > 1,
  N1 is N+1,
  clique(N,_),
  \+(clique(N1,_)),
  !,
  write('..looking for cliques of size '), write(N1), nl,
  enlarge_cliques(N,N1).

%%% Find two cliques of size N that differ only in their FIRST element.
%%% If there is also an edge between those two first elements, we have
%%% found a larger clique. ALWAYS KEEP CLIQUES IN A STANDARD FORM: a
%%% sorted list, smallest first.
%%%    In particular, cliques of size 2 are just edges, so we must
%%% rely on the fact that in any edge fact edge(P,Q), P is less than Q.
%%% If your data is not of this form, add an interface to it, eg
%%%   edge(A,B) :-
%%%     ( actual_edge(A,B)
%%%     ; actual_edge(B,A)
%%%     ), A < B.

enlarge_cliques(N,M) :-
  clique(N,[A|Rest]),
  clique(N,[B|Rest]),
  A < B,
  edge(A,B),
  assertz(clique(M,[A,B|Rest])), fail.
enlarge_cliques(_,_).

%%% Cliques of size 2 are just edges.

clique(2,[A,B]) :-
  edge(A,B),
  A < B.


%%% Test data:

edge(0 , 1).
edge(0 , 2).
edge(0 , 8).
edge(0 , 9).
edge(0 , 11).
edge(0 , 12).
edge(0 , 13).
edge(0 , 14).
edge(0 , 15).
edge(0 , 16).
edge(0 , 19).
edge(0 , 20).
edge(0 , 21).
edge(0 , 22).
edge(0 , 24).
edge(0 , 25).
edge(0 , 26).
edge(0 , 27).
edge(0 , 28).
edge(0 , 29).
edge(1 , 2).
edge(1 , 8).
edge(1 , 9).
edge(1 , 11).
edge(1 , 12).
edge(1 , 13).
edge(1 , 14).
edge(1 , 15).
edge(1 , 16).
edge(1 , 19).
edge(1 , 20).
edge(1 , 21).
edge(1 , 22).
edge(1 , 24).
edge(1 , 25).
edge(1 , 26).
edge(1 , 27).
edge(1 , 28).
edge(1 , 29).
edge(2 , 8).
edge(2 , 9).
edge(2 , 11).
edge(2 , 12).
edge(2 , 13).
edge(2 , 14).
edge(2 , 15).
edge(2 , 16).
edge(2 , 19).
edge(2 , 20).
edge(2 , 21).
edge(2 , 22).
edge(2 , 24).
edge(2 , 25).
edge(2 , 26).
edge(2 , 27).
edge(2 , 28).
edge(2 , 29).
edge(3 , 4).
edge(3 , 5).
edge(3 , 6).
edge(3 , 7).
edge(3 , 11).
edge(3 , 12).
edge(3 , 24).
edge(3 , 25).
edge(4 , 5).
edge(4 , 6).
edge(4 , 7).
edge(4 , 11).
edge(4 , 12).
edge(4 , 24).
edge(4 , 25).
edge(5 , 6).
edge(5 , 7).
edge(5 , 11).
edge(5 , 12).
edge(5 , 24).
edge(5 , 25).
edge(6 , 7).
edge(6 , 11).
edge(6 , 12).
edge(6 , 13).
edge(6 , 14).
edge(6 , 15).
edge(6 , 16).
edge(6 , 17).
edge(6 , 18).
edge(6 , 21).
edge(6 , 22).
edge(6 , 24).
edge(6 , 25).
edge(6 , 27).
edge(6 , 28).
edge(6 , 29).
edge(7 , 11).
edge(7 , 12).
edge(7 , 13).
edge(7 , 14).
edge(7 , 15).
edge(7 , 16).
edge(7 , 17).
edge(7 , 18).
edge(7 , 21).
edge(7 , 22).
edge(7 , 24).
edge(7 , 25).
edge(7 , 27).
edge(7 , 28).
edge(7 , 29).
edge(8 , 10).
edge(8 , 11).
edge(8 , 12).
edge(8 , 15).
edge(8 , 16).
edge(8 , 21).
edge(8 , 22).
edge(8 , 23).
edge(8 , 26).
edge(8 , 27).
edge(8 , 28).
edge(8 , 29).
edge(9 , 19).
edge(9 , 20).
edge(10, 23).
edge(11, 12).
edge(11, 15).
edge(11, 16).
edge(11, 17).
edge(11, 18).
edge(12, 15).
edge(12, 16).
edge(12, 17).
edge(12, 18).
edge(13, 14).
edge(13, 21).
edge(13, 22).
edge(13, 24).
edge(13, 25).
edge(13, 27).
edge(13, 28).
edge(13, 29).
edge(14, 21).
edge(14, 22).
edge(14, 24).
edge(14, 25).
edge(14, 27).
edge(14, 28).
edge(14, 29).
edge(15, 16).
edge(15, 21).
edge(15, 22).
edge(15, 27).
edge(15, 28).
edge(15, 29).
edge(16, 21).
edge(16, 22).
edge(16, 27).
edge(16, 28).
edge(16, 29).
edge(17, 18).
edge(17, 21).
edge(17, 22).
edge(18, 21).
edge(18, 22).
edge(19, 20).
edge(19, 21).
edge(19, 22).
edge(20, 21).
edge(20, 22).
edge(21, 22).
edge(21, 27).
edge(21, 28).
edge(21, 29).
edge(22, 27).
edge(22, 28).
edge(22, 29).
edge(24, 25).
edge(27, 29).
edge(28, 29).

